(ns simply.messaging
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [simply.cqrs :as cqrs]
            [simply.spec.whatsapp :as spec.whatsapp]))

;;;; LEADS

(def lead-topic "leads")

(defmethod cqrs/handle-action :leads/send [t leads {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t leads
   (fn []
     (cqrs/send-messages cqrs-system lead-topic leads))))

(defn send-leads-action [leads]
  (cqrs/action :leads/send leads))

(defn send-lead-action [lead]
  (send-leads-action [lead]))

;;;; BROKERS

(def brokers-topic "brokers")

(defn broker-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  {:type type :payload payload})

(defmethod cqrs/handle-action :send-broker-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system brokers-topic msgs))))

(defn send-broker-messages-action [msgs]
  (cqrs/action :send-broker-messages msgs))

(defn send-broker-message-action [type payload]
  (send-broker-messages-action [(broker-message type payload)]))

;;;; LETTERS

(def letters-topic "letters")

(defn letter-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  {:type type :payload payload})

(defmethod cqrs/handle-action :send-letter-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system letters-topic msgs))))

(defn send-letter-messages-action [msgs]
  (cqrs/action :send-letter-messages msgs))

(defn send-letter-message-action [type payload]
  (send-letter-messages-action [(letter-message type payload)]))

;;;; COMPANIES

(def companies-topic "companies")

(defn company-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  {:type type :payload payload})

(defmethod cqrs/handle-action :send-company-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system companies-topic msgs))))

(defn send-company-messages-action [msgs]
  (cqrs/action :send-company-messages msgs))

(defn send-company-message-action [type payload]
  (send-company-messages-action [(company-message type payload)]))

;;;; CONTRACTS

(def contracts-topic "contracts")

(defn contracts-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  (assoc payload :type type))

(defmethod cqrs/handle-action :send-contracts-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system contracts-topic msgs))))

(defn send-contracts-messages-action [msgs]
  (cqrs/action :send-contracts-messages msgs))

(defn send-contracts-message-action [type payload]
  (send-contracts-messages-action [(contracts-message type payload)]))

;;;; SMS

(defn sms
  "* type is a string defining the sms type
   * reference is a string reference to the entity"
  ([type number text reference]
   {:type type :number number :text text :reference reference})
  ([type number text]
   {:type type :number number :text text}))

(def sms-topic "sms")

(defmethod cqrs/handle-action :send-smses [t smses {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t smses
   (fn []
     (cqrs/send-messages cqrs-system sms-topic smses))))

(defn send-multiple-smses-action [& smses]
  (cqrs/action :send-smses smses))

(defn send-sms-action [sms]
  (send-multiple-smses-action sms))

;;;; WHATSAPP

(defn validate-whatsapp-message-set
  "Validates a message-set to be sent via the :send-whatsapp-messages action.
  See Postaldistrix-2 README for more info, also specs at `simply.spec.whatsapp`"

  [message-set]
  (if-not (s/valid? ::spec.whatsapp/message message-set)
    (throw (ex-info "Malformed Whatsapp Message Set"
                    (s/explain-data ::spec.whatsapp/message message-set)))
    message-set))

(def whatsapp-topic "whatsapp")

(defmethod cqrs/handle-action :send-whatsapp-messages [t messages {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t messages
   (fn []
     (cqrs/send-messages cqrs-system sms-topic messages))))

;;;; Marketing

(def marketing-topic "marketing")

(defn marketing-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  (assoc payload :type type))

(defmethod cqrs/handle-action :send-marketing-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system marketing-topic msgs))))

(defn send-marketing-messages-action [msgs]
  (cqrs/action :send-marketing-messages msgs))

(defn send-marketing-message-action [type payload]
  (send-marketing-messages-action [(marketing-message type payload)]))

;;;; EMAILS

(defn email
  "* type is a string defining the email type
   * to can be a string or vec of strings
   * from needs to be a mailjet from email address
   * body can be text or html. If html then html? should be true
   * reference is a string reference to the entity
   * attachements is an vec of email-attachement"
  ([type to from subject body]
   (email type to from subject body false nil []))
  ([type to from subject body html?]
   (email type to from subject body html? nil []))
  ([type to from subject body html? reference]
   (email type to from subject body html? reference []))
  ([type to from subject body html? reference attachments]
   (let [to (if (string? to) [to] to)]
     {:type type :reference reference
      :from from
      :to to :subject subject :body body :is-html html?
      :attachments attachments})))

(defn email-attachment
  [mime-type name byte-array]
  {:content (String. (b64/encode byte-array))
   :type    mime-type
   :name    name})

(def email-topic "email")

(defmethod cqrs/handle-action :send-emails [t emails {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t emails
   (fn []
     (cqrs/send-messages cqrs-system email-topic emails))))

(defn send-multiple-emails-action [& emails]
  (cqrs/action :send-emails emails))

(defn send-email-action [email]
  (send-multiple-emails-action email))

(comment
  "Example Email"
  (cqrs/command-result
   [(cqrs/event :foo {})]
   [(send-email-action (email "jou ma se mail" "a@b.com" "this-should-be-a-mailjet@b.com" "subject" "text-body" false "some ref number"))])

  "Example attachment (how you get the byte array is up to you)"
  (email-attachment "application/pdf"
                    "name.pdf"
                    (.toByteArray some-stream-stream)))

;;;; SLACK

(defn slack
  [token channel type text]
  {:token token :channel channel :type type :text text})

(def slack-topic "slack")

(defmethod cqrs/handle-action :send-slacks [t slacks {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t slacks
   (fn []
     (cqrs/send-messages cqrs-system slack-topic slacks))))

(defn send-multiple-slacks-action [& slacks]
  (cqrs/action :send-slacks slacks))

(defn send-slack-action [slack]
  (send-multiple-slacks-action slack))

;; Zendesk integration

(defn zendesk-ticket [type subject body reference contract-id tags sender-reference]
  {:subject subject
   :type type
   :body body
   :reference reference
   :contract-id contract-id
   :tags tags
   :sender-reference sender-reference})

(def zendesk-topic "zendesk-integration")

(defmethod cqrs/handle-action :zendesk/create-tickets
  [action-type tickets {:keys [cqrs cqrs-system]}]

  (cqrs/chunk-events
   cqrs action-type tickets
   (fn []
     (let [messages (map (fn [ticket]
                           {:type "CREATE-TICKET"
                            :ticket ticket})
                         tickets)]
       (cqrs/send-messages cqrs-system
                           "zendesk-integration"
                           messages)))))

;; NEW CONTRACTS

(def new-contract-topic "new-contracts")

(defmethod cqrs/handle-action :new-contracts/send [t contracts {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t contracts
   (fn []
     (cqrs/send-messages cqrs-system new-contract-topic contracts))))

(defn send-new-contracts-action [& contracts]
  (cqrs/action :new-contracts/send contracts))

(defn send-new-contract-action [contract]
  (send-new-contracts-action contract))
