(ns simply.gcp.logging
  (:require [cheshire.core :as cheshire]
            [clj-time.format :as time.format]
            [clj-time.coerce :as time.coerce]
            [taoensso.timbre :as log]
            [simply.http.client :as http]
            [simply.gcp.util :as util]
            [integrant.core :as ig]))

(defn wrap-client [client]
  (fn [config]
    (client
     (merge
      config
      {:url (format "https://logging.googleapis.com/v2/%s" (:url config))}
      {:scope "https://www.googleapis.com/auth/logging.write"}))))


(defn write-log-request [params]
  {:method :post
   :url    "entries:write"
   :body   (cheshire/generate-string params)})


(def level-mapping
  ;; see https://cloud.google.com/logging/docs/api/reference/rest/v2/LogEntry#LogSeverity
  {:debug "DEBUG"
   :info  "INFO"
   :warn  "WARNING"
   :error "ERROR"
   :fatal "CRITICAL"})


(defn prepare-log-entry-payload [timbre-log-data]
  (let [{:keys [vargs msg_ ?err ?file hostname_ ?ns-str ?line]} timbre-log-data]
    {:vargs_str (pr-str vargs)
     :msg       @msg_
     :err       (pr-str ?err)
     :file      ?file
     :hostname  @hostname_
     :ns_str    ?ns-str
     :line      ?line}))


(defn prepare-log-entry [timbre-log-data]
  ;; see https://cloud.google.com/logging/docs/api/reference/rest/v2/LogEntry
  {:timestamp (util/unparse-zulu-date-format (:instant timbre-log-data))
   :severity (get level-mapping (:level timbre-log-data) "DEFAULT")
   :insertId (str (.getTime (:instant timbre-log-data)) "_" @(:hash_ timbre-log-data))
   :jsonPayload (prepare-log-entry-payload timbre-log-data)})


(defn appender-fn [client context]
  (fn [log-data]
    (when (not= (:context log-data) ::appending)
      (try
        (log/with-context
          ::appending
          (client
           (write-log-request
            {:entries
             [(merge context
                     (prepare-log-entry log-data))]})))
        (catch Exception e)))))
          ;; ignore since logging could cause an endless loop here, if
          ;; the appending code also writes something to the log (over
          ;; timbre, slf4j etc.).


;;;; Metadata

(defn extract-zone [zone-str]
  (second (re-find #".*/zones/(.*)" zone-str)))

(defn extract-project-id [hostname]
  (second (re-find #".*\.(.*?)\.internal$" hostname)))

(defn prepare-log-metadata [container-name container-id instance-metadata]
  (let [project-id (extract-project-id (:hostname instance-metadata))
        {:keys [cluster-name ]} (:attributes instance-metadata)]
    {:logName (str "projects/"
                   project-id
                   "/logs/"
                   container-name)
     :resource {:type "container"
                :labels
                {:zone (extract-zone (:zone instance-metadata))
                 :instance_id (str (:id instance-metadata))
                 :cluster_name cluster-name
                 :container_name container-name
                 :pod_id (str container-id)
                 :project_id project-id}}}))

(def instance-metadata-request
  {:method :get
   :url "http://metadata.google.internal/computeMetadata/v1/instance/"
   :query-params {:recursive true}
   :headers {"Metadata-Flavor" "Google"}})


(defn get-entry-metadata*
  "Only works on a Google Compute Engine (GCE) instance (since
   http://metadata.google.internal/ is only available there)"
  [request-fn]
  (let [response (request-fn instance-metadata-request)
        container-name (or (System/getenv "CONTAINER_NAME") "tenandsix")
        container-id (or (System/getenv "CONTAINER_ID") "tenandsix")]
    (prepare-log-metadata container-name container-id (:body response))))


(defn get-entry-metadata []
  (get-entry-metadata*
   (fn [req]
     (let [response @(http/request req)]
       (update
        response
        :body
        #(cheshire/parse-string % true))))))


;;;; Log Appender

(def test-meta
  {:logName "projects/simply-pre-prod/logs/stdout"
   :resource
   {:labels
    {;:cluster_name   "asterix-pre-prod"
     ;:container_name "tenandsix-pre-prod"
     ;:instance_id    "2783793341617253296"
     ;:namespace_name   "default"
     ;:pod_name         "tenandsix-pre-prod-765ff548c4-k4k4x"
     :project_id     "simply-pre-prod"
     :location           "us-east1-b"}
    :type "k8s_container"}})


(defn appender
  "returns a timbre appender"
  [{:keys [client]}]
  (let [context (get-entry-metadata)]
    {:enabled?   true
     :async?     true
     :min-level  :info
     :rate-limit nil
     :output-fn  :inherit
     :fn         (appender-fn (wrap-client client) context)}))


(defmethod ig/init-key :simply.gcp.logging/appender
  [_ options]
  {:name :cloudlogging
   :appender (appender options)})
