(ns simply.helpers
  (:require
   [simply.gcp.pubsub.core :as p]
   [simply.gcp.pubsub.cqrs :as cqrs.ps]
   [simply.cqrs :as cqrs]
   [simply.gcp.auth :as auth]
   [simply.gcp.keys :as s.keys]
   [simply.gcp.persistence.db :as ps.db]))


(def ^:private envs
  {:emulator [:emulator "dogmatix" #(auth/emulator-client) #(p/emulator-client "8321" "dogmatix" (auth/emulator-client))]
   :prod [:simply-prod "simply-prod"]
   :pre-prod [:simply-pre-prod "simply-pre-prod"]
   :om [:old-mutual-prod "old-mutual-prod"]
   :absa [:absa-prod "absa-prod"]
   :rml [:rml-prod "rml-prod"]
   :partner [:partner-preprod "partner-preprod"]})


(defn available-envs []
  (keys envs))


(defn- guard-config [env]
  (when-not (contains? envs env)
    (throw (ex-info "Invalid Env" {:env env
                                   :available-envs (available-envs)}))))


(defn ->project-id [env]
  (guard-config env)
  (get-in envs [env 1]))


(defn ->client [env]
  (guard-config env)
  (auth/reset-credentials!)
  (let [[k _ client] (get envs env)]
    (if client
      (client)
      (auth/keyfile-client (s.keys/key-file k)))))


(defn ->pubsub-client [env]
  (guard-config env)
  (auth/reset-credentials!)
  (let [[k p-id _ pubsub-client] (get envs env)]
    (if pubsub-client
      (pubsub-client)
      (p/production-client p-id (auth/keyfile-client (s.keys/key-file k))))))


(defn trigger-action [service env action-key data & {:keys [print-data?] :or {print-data? true}}]
  (let [client (->pubsub-client env)
        action (cqrs/action action-key data)
        topic-key (cqrs.ps/required-actions-topic-key service)
        topic (cqrs.ps/required-actions-topic topic-key)
        pubsub (p/pubsub client [topic] [])]
    (auth/reset-credentials!)
    (p/init pubsub)
    (prn (str "Requesting " action-key))
    (when print-data?
      (clojure.pprint/pprint data))
    (p/publish pubsub topic-key [action])
    (prn "Done")))


(defn trigger-command [& {:keys [service env command data user-id print-data?] :or {print-data? true}}]
  (trigger-action service env
                  :helpers/handle-command
                  {:command-type command
                   :data data
                   :user-id user-id}
                  :print-data? print-data?))


(defmacro with-db
  [env & body]
  `(let [project-id# (simply.helpers/->project-id ~env)
         client# (simply.helpers/->client ~env)
         db# (simply.gcp.persistence.db/db
             {:project-id project-id#
              :client client#})]
     (with-redefs
       [simply.persistence.core/->db (constantly db#)]
       ~@body)))
