(ns simply.deps
  (:require [integrant.core :as ig]
            [simply.errors :as e]
            [simply.cqrs :as cqrs]
            [clojure.string :as string]
            [taoensso.timbre :as logger]))

(defonce ^:private *dependencies (atom {}))

(defonce ^:private *predefined-dependencies (atom {}))

(defn set-predefined-dependency! [k f]
  (swap! *predefined-dependencies assoc k f))

(defn- get-predef-dep [k]
  (when-let [f (get @*predefined-dependencies k)]
    (f)))


(defn get-dep
  [k & {:keys [satisfies]}]
  (let [dep (or (get @*dependencies k)
                (get-predef-dep k))]

    (when (not (some? dep))
      (e/throw-app-error
       "Dependency not defined, ensure integrant is started and the simply.deps/deps key is added to your config and has the :dep key set"
       {:dep k}))

    (when (and satisfies (not (satisfies? satisfies dep)))
      (e/throw-app-error
       "Dependency does not satisfy expected protocol"
       {:dep k
        :expects (str (:on satisfies))
        :actual (type dep)}))

    dep))


(defn- cqrs-system []
  (if (true? (:simply.deps/do-not-check-cqrs-system-satisfies-protocol @*dependencies))
    (get-dep :cqrs-system)
    (get-dep :cqrs-system :satisfies cqrs/CqrsSystem)))


(defn send-messages [topic coll]
  (cqrs/send-messages (cqrs-system) topic coll))


(defn send-message [topic message]
  (send-messages topic [message]))


(defn handle-command [type data user]
  (let [handle (cqrs/get-command-handler (cqrs-system))]
    (handle type data user)))


(defn require-actions [actions]
  (cqrs/request-actions (cqrs-system) actions))


(defn set-deps! [deps-map]
  (reset! *dependencies deps-map))


(defmethod ig/init-key :simply.deps/deps [_ dependencies]
  (set-deps! dependencies))


(defmethod ig/halt-key! :simply.deps/deps [_ _]
  (set-deps! {}))


;;MEH
;; I've placed this here because simply.deps will get imported into existing apps,
;; meaning I don't have to go to every app and import this manually
(defmethod cqrs/handle-action :helpers/handle-command
  [_ {:keys [user-id command-type data]} _]
  (let [errors (->> [(when-not (contains? (methods cqrs/handle-command) command-type)
                       "Invalid command-type")
                     (when-not (and (string? user-id) (not (string/blank? user-id)))
                       "Invalid user-id")
                     (when-not (map? data)
                       "Invalid command-data")]
                    (remove nil?))
        valid? (empty? errors)]
    (if valid?
      (try
        (handle-command command-type data (cqrs/cqrs-user user-id))
        (catch Exception e
          (logger/error (ex-info "::handle-command action failed during processing"
                                 {:user-id user-id
                                  :command-type command-type} e))))
      (logger/error (ex-info "::handle-command action failed"
                             {:user-id user-id
                              :command-type command-type
                              :errors errors})))))
