(ns simply.web
  (:require [compojure.core :refer [context GET POST routes]]
            [compojure.route :as route]
            [ring.util.response :as response]
            [clojure.string :as string]
            [taoensso.timbre :as log]))


(defn google-tag-manager-scripts
  "Generates Google Tag Manager scripts if `GOOGLE_TAG_MANAGER_ID` env variable is available.
   returns {:script \"...\" :no-script \"...\"}"
  []
  (let [tag-id (System/getenv "GOOGLE_TAG_MANAGER_ID")]
    (if-not tag-id
      {:script ""
       :no-script ""}
      {:script
       (format
        "<script>
          dataLayer = [];
         </script>
         <!-- Google Tag Manager -->
          <script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
          new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
          j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
          'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
          })(window,document,'script','dataLayer','%s');</script>
         <!-- End Google Tag Manager -->"
        tag-id)
       :no-script
       (format
        "<!-- Google Tag Manager (noscript) -->
           <noscript><iframe src=\"https://www.googletagmanager.com/ns.html?id=%s\"
           height=\"0\" width=\"0\" style=\"display:none;visibility:hidden\"></iframe></noscript>
         <!-- End Google Tag Manager (noscript) -->"
        tag-id)})))


(defn google-cloud-project-routes
  "Routes for GCP container health checks"
  []
  (routes
   (GET "/_ah/health" [] "Healthy")
   (GET "/_ah/background" [] "Healthz")
   (GET "/healthz" [] "Healthy")))


(defn debug-routes
  "Routes for checking versions and testing exceptions"
  [{:keys [version] :as config}]
  (routes
   (GET "/v" [] version)
   (GET "/version" [] version)
   (GET "/throw" [] (throw (Exception. "TEST")))))


(defn wrap-https-redirect
  [redirect-http? handler]
  (fn [{:keys [scheme server-name uri query-string request-method] :as request}]
    (if (and (true? redirect-http?) (= :http scheme))
      (response/redirect (str "https://"
                              server-name
                              (if (string/ends-with? uri "/")
                                (-> uri drop-last string/join)
                                uri)
                              (if query-string
                                (str "?" query-string) ""))
                         301)
      (handler request))))
