(ns simply.astronomix.assets
  (:require [clojure.java.io :as io]
            [clojure.string :as string]
            [hickory.core :as hickory]
            [hickory.select :as select]
            [me.raynes.fs :as fs]))


(def ^:private endpoint "https://unpkg.com/@simply-fin-services/astronomix@")


(defn- ->asset-path [version]
  (str endpoint version "/umd/"))


(defn- ->js-path [version]
  (str (->asset-path version) "@simply-fin-services/"))


(defn- fetch-page-as-hickory
  [path]
  (->> path io/as-url slurp hickory/parse hickory/as-hickory))


(defn- extract-table-links [hickory-data]
  (->>
   (select/select (select/child (select/tag :tbody)
                                (select/tag :tr)
                                (select/tag :td)
                                (select/tag :a))
                  hickory-data)
   (map #(get-in % [:attrs :href]))
   (filter #(not (string/ends-with? % "/")))))


(defn- download-asset [source destination]
  (let [dest-file (io/file destination)]
    (io/make-parents dest-file)
    (with-open [in (io/input-stream (io/as-url source))
                out (io/output-stream dest-file)]
      (io/copy in out))))


(defn with-retries [n f]
  (try
    (f)
    (catch Exception e
      (if (> n 0)
        (do
          (prn (str "FAIL! RETRIES LEFT -> " n " " (:cause (Throwable->map e))))
          (with-retries (dec n) f))
        (throw e)))))


(defn- download-assets-from-page [source destination]
  (let [assets (->> source
                    fetch-page-as-hickory
                    extract-table-links)]
    (doseq [asset assets]
      (with-retries 10
        (fn []
          (prn (str "Downloading: " asset))
          (download-asset (str source asset)
                          (str destination asset))
          (Thread/sleep 1000))))
    (prn "DONE!")))


(defn- add-slash [s]
  (if (string/ends-with? s "/")
    s
    (str s "/")))


(defn download [version asset-dir js-dir]
  (let [asset-dir (add-slash asset-dir)
        js-dir (add-slash js-dir)]
    (fs/delete-dir (io/file asset-dir))
    (fs/delete-dir (io/file js-dir))
    (download-assets-from-page (->asset-path version)  asset-dir)
    (download-assets-from-page (->js-path version)  js-dir)
    (spit (str (add-slash js-dir) "astro-version.txt") version)))


(defn -main [[version asset-dir js-dir & args]]
  (download version asset-dir js-dir))
