(ns simply.gcp.datastore.core
  (:require [cheshire.core :as json]
            [simply.gcp.datastore.entities :as entities]
            [simply.errors :as e]
            [integrant.core :as ig]
            [camel-snake-kebab.extras :as camel-snake.extras]
            [camel-snake-kebab.core :as camel-snake]
            [clojure.string :as str]))

;; entities and queries

(defn wrap-query-params
  "Returns a function that takes query parameters
   and adds project-id and namespace to the query-params."
  [project-id namespace]
  (fn [query-params]
    (merge query-params {:ds/namespace-id namespace
                  :ds/project-id project-id})))


(defn wrap-entity
  "Returns a function that takes an entity
   and adds project-id and namespace on the entity."
  [project-id namespace]
  (fn [entity]
    (merge entity {:ds/namespace-id namespace
                   :ds/project-id project-id})))


(defn unwrap-entity
  "Removes datastore keys from the entity"
  [entity]
  (dissoc entity :ds/kind :ds/indexes :ds/ancestor :ds/namespace-id :ds/name :ds/id))


(defn- with-clojure-keys [m]
  (camel-snake.extras/transform-keys camel-snake/->kebab-case-keyword m))


(defn domain-entity
  "turns a datastore entity into a domain entity removing datastore keys"
  [m]
  (-> m entities/parse-entity unwrap-entity with-clojure-keys))


(defn name-key [kind name]
  {:ds/kind kind :ds/name name})


(defn db-uuid [v] (if (uuid? v)
                    (.toString v)
                    v))


(defn app-uuid [v] (if (string? v)
                     (try (java.util.UUID/fromString v)
                          (catch IllegalArgumentException _ v))
                     v))


(defn db-date [v] (if (number? v)
                    (java.util.Date. v)
                    v))


(defn app-date [v] (try (.getTime v)
                        (catch IllegalArgumentException _ v)))


(def entity-help
  "* Use wrap-entity to add project-id and namespace
   * Entity needs the following keys
      - :ds/kind  String
      - either :ds/name  String    or    :ds/id  Int
      - :ds/indexes #{<keys of fields that needs to be indexed>}")

;; api


(def start-transaction-request-params
  {:method :post
   :url ":beginTransaction"})


(defn- decode [e] (json/parse-string e keyword))


(defn start-transaction [client]
  (-> start-transaction-request-params
      client
      :body
      decode
      :transaction))


;; Lookup

(defn- lookup-request [params]
  {:method :post
   :url ":lookup"
   :body
   {:readOptions
    (if-let [transaction (:transaction params)]
      {:transaction transaction}
      {:readConsistency (:readConsistency params "STRONG")})
    :keys (:keys params)}})


(defn- lookup-entity-request [entity]
  (let [key (entities/prepare-key entity)]
    (lookup-request
     (merge (select-keys entity [:transaction :readConsistency])
            {:keys [key]}))))


(defn lookup-entity
  "Fetches and Parses a datastore Entity. If not found returns nil.
   * Use the api-client fn to construct an api-client.
   * Use name-key to construct the entity key."
  [api-client entity-key]
  (let [response (api-client
                  (lookup-entity-request
                   entity-key))
        body (decode (:body response))]
    (when-let [entity (get-in body [:found 0])]
      (entities/parse-entity entity))))


;; Change

(def transactional-mode "TRANSACTIONAL")
(def non-transactional-mode "NON_TRANSACTIONAL")


(defn commit-request [params]
  {:method :post
   :url ":commit"
   :body
   (merge
    {:mode (:mode params)
     :mutations (:mutations params)}
    (when-let [transaction (:transaction params)]
      {:transaction transaction}))})


(defn upsert-mutation [entity] {:upsert (entities/prepare-entity entity)})


(defn- upsert-entities-request [params entity-coll]
  (commit-request
   (merge
    params
    {:mutations (->> entity-coll (map upsert-mutation))})))


(defn- upsert-entity-request [entity]
  (upsert-entities-request {:mode non-transactional-mode} [entity]))


(defn upsert-entity
  "Creates or updates an entity.
   * Use the api-client fn to construct an api-client.
   * See entity-help for info regarding entities"
  [api-client entity]
  (-> entity
      upsert-entity-request
      api-client))


(defn upsert-entities
  "Creates or updates entities in a transaction.
   * Use the api-client fn to construct an api-client.
   * See entity-help for info regarding entities"
  [api-client entities]
  (let [transaction (start-transaction api-client)
        request (upsert-entities-request
                 {:mode transactional-mode
                  :transaction transaction}
                 entities)]
    (api-client request)))


(defn commit-mutations [api-client mutations]
  (let [transaction (start-transaction api-client)
        request (commit-request {:mode transactional-mode
                                 :transaction transaction
                                 :mutations mutations})]
    (api-client request)))


;; query

(def strong-read {:readConsistency "STRONG"})
(def weak-read {:readConsistency "EVENTUAL"})
(def strong-read-option {:read-option strong-read})
(def weak-read-option {:read-option weak-read})


(defn- run-query-request [query params]
  {:method :post
   :url ":runQuery"
   :body
   {:partitionId (entities/prepare-partition params)
    :gqlQuery {:queryString query
               :allowLiterals true}}})


(defn- run-query-next-page-request [query params cursor]
  {:method :post
   :url ":runQuery"
   :body
   {:partitionId (entities/prepare-partition params)
    :gqlQuery {:allowLiterals true
               :namedBindings {:cursor {:cursor cursor}}
               :queryString (str query " OFFSET @cursor")}}})


(defn run-query
  "Runs a qgl query
  * Use the api-client fn to construct an api-client.
  * Use wrap-query-params to add project-id and namespace to query-params
  * Pass in an optional limit up to 300 (default 300)"
  ([api-client gql-query query-params]
   (run-query api-client gql-query query-params 300))
  ([api-client gql-query query-params limit]
   (when (.contains (.toUpperCase gql-query) "LIMIT")
     (e/throw-app-error "run-query should not have LIMIT spesified as part of the gql-query" {:query gql-query}))
   (when (> limit 300)
     (e/throw-app-error "run-query limit cannot exceed 300"))
   (when (> 1 limit)
     (e/throw-app-error "run-query limit should be more than 0"))
   (let [gql-query (str gql-query " LIMIT " limit)]
     (loop [request (run-query-request gql-query query-params)
            result []]
       (let [{:keys [batch query]} (-> request
                                       api-client
                                       :body
                                       decode)
             {:keys [moreResults endCursor entityResults]} batch
             combined-results (concat result entityResults)]
         (if (= moreResults "MORE_RESULTS_AFTER_LIMIT")
           (recur
            (run-query-next-page-request gql-query query-params endCursor)
            combined-results)
           combined-results))))))



;; Base

(defn- wrap-datastore-params [project-id http-client]
  (fn [config]
    (cond-> config
        true (update :url #(format "https://datastore.googleapis.com/v1/projects/%s%s" project-id %))
        true (assoc :scope "https://www.googleapis.com/auth/datastore")
        (contains? config :body) (update :body json/generate-string)
        true http-client)))


(defn- wrap-error-handling [client]
  (fn [config]
    (let [{:keys [status body error opts] :as response} @(client config)]
      (when (or (nil? status) (<= 300 status) (> 200 status))
        (e/throw-app-error "Datastore Request Failed" {:status status
                                                       :body   body
                                                       :error  error
                                                       :url    (:url opts)
                                                       :method (:method opts)}))
      response)))


(defn api-client
  "wraps a simple.gcp.auth client with datastore parameters and error handling"
  [{:keys [project-id client]}]
  (->> client
       (wrap-datastore-params project-id)
       wrap-error-handling))
