(ns simply.wording-config-parser
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.edn :as edn]
            [clojure.walk :refer [postwalk]]
            [clojure.string :as string]
            [simply.core :as c]
            [com.walmartlabs.dyn-edn :as dyn-edn]))


;;;; READERS

(defprotocol WordingReference
  (path [this]))


(defn- wording-reference [v]
  (reify WordingReference
    (path [this] v)))


(s/def ::reference-path
  (s/or ::keyword keyword?
        ::keyword-vec (s/coll-of keyword? :kind vector? :min-count 1)))


(defn reference-reader
  [v]
  (s/assert ::reference-path v)
  (let [[t v] (s/conform ::reference-path v)]
    (if (= ::keyword t)
      (wording-reference [v])
      (wording-reference v))))


(defn wording-readers []
  (merge
   (dyn-edn/env-readers)
   {'sw/ref reference-reader}))


;;;; REFERENCES

(defn resolve-references-based-on-external-config [config o]
  (postwalk
   (fn [v]
     (loop [v v]
       (if (satisfies? WordingReference v)
         (let [v' (path v)
               config-v (get-in config v')
               o-v (get-in o v')]
           (recur (or config-v o-v)))
         v)))
   o))

(defn resolve-references
  [types o]
  (loop [types types
         o o]
    (if (empty? types)
      o
      (recur
       (rest types)
       (update o (first types) #(resolve-references-based-on-external-config o %))))))


;;;; READ EDN

(defn read-edn-file [file]
  (println (str "reading config file: " file))
  (edn/read-string {:readers (wording-readers)} (slurp file)))


(defn throw-config-file-not-found [file-path
                                   config-type]
  (let [error-message (str "No such wording config file in ./resources directory: "
                           file-path " for wording type: `" config-type "`. "
                           "Please ensure that an edn file is present for each wording config type.")]
    (throw (Exception. error-message))))


(defn config-file-path [config-directory-name
                        partner
                        config-type]
  (str config-directory-name "/"
       partner "/" (name config-type) ".edn"))


(defn load-config
  ([config-directory-name
    partner
    types]

   (load-config config-directory-name
                partner
                types
                {:throw-exception-if-no-config-file? true}))

  ([config-directory-name
    partner
    types
    {:keys [throw-exception-if-no-config-file?] :as options}]

   (into {}
         (map (fn [config-type]

                (let [file-path           (config-file-path config-directory-name
                                                            partner
                                                            config-type)

                      file                (io/resource file-path)

                      config-file-exists? (not (nil? file))]

                  (if config-file-exists?
                    {config-type (read-edn-file file)}
                    (when throw-exception-if-no-config-file?
                      (throw-config-file-not-found file-path
                                                   config-type)))))
              types))))


(def ^:private default-base-config "base")
(def ^:private default-config-directory "wording-config")

(defn parse-config [& {:keys [partner types base-config-name config-directory]
                       :or {types [:all]
                            base-config-name default-base-config
                            config-directory default-config-directory}}]

  (let [base-config              (load-config config-directory
                                              default-base-config
                                              types
                                              {:throw-exception-if-no-config-file? false})

        partner-config           (load-config config-directory
                                              partner
                                              types
                                              {:throw-exception-if-no-config-file? false})

        effective-config         (merge-with merge
                                             base-config
                                             partner-config)]
    (resolve-references types effective-config)))


;; TESTING

(defn- edn-file? [f]
  (and (.isFile f) (string/ends-with? (.toString f) ".edn")))


(defn- get-all-types-in-dir
  "Extracts all the wording types from a given directory"
  [dir]
  (->> (io/file dir)
       file-seq
       (filter edn-file?)
       (map #(-> %
                 .toString
                 (string/replace-first  dir "")
                 (string/replace ".edn" "")
                 keyword))
       set))


(defn- get-partners
  "extract all partner keys from a given directory"
  [config-directory base-config-name]
  (let [d       (str "resources/" config-directory "/")
        folders (->> (io/file d)
                     (file-seq)
                     (drop 1)
                     (filter #(.isDirectory %))
                     (map #(-> % .toString (string/replace d "")))
                     set)]
    (disj folders base-config-name)))


(defn get-all-types-for-partner
  "Returns a set containing all the config types defined for a partner"
  ([& {:keys [partner base-config-name config-directory]
       :or {base-config-name default-base-config
            config-directory default-config-directory}}]

   (let [->types (fn [p] (get-all-types-in-dir (str "resources/" config-directory "/" p "/")))]

     (clojure.set/union
      (->types base-config-name)
      (->types partner)))))


(defn partner-config-mismatch
  "returns a list of partners that are missing wording config keys
   If partners are not defined, it will use all provided partners in the config-directory
   If types are not defined, it will use the types for simply, or for the first partner if simply is not provided
  "
  [& {:keys [partners base-config-name config-directory types]
                                  :or {base-config-name default-base-config
                                       config-directory default-config-directory}}]
  (let [partners (or partners (get-partners config-directory base-config-name))]
    (if (< (count partners) 2)
      []
      (let [partners (set partners)

            compare-partner (or (partners "simply") (first partners))

            compare-partners (disj partners compare-partner)

            types (or types (get-all-types-for-partner :partner compare-partner
                                                       :base-config-name base-config-name
                                                       :config-directory config-directory))

            reference-config (-> (parse-config :partner compare-partner
                                               :base-config-name base-config-name
                                               :config-directory config-directory
                                               :types types)
                                 c/keys-in
                                 set)]

        (->> compare-partners

             (map (fn [p]
                    [p (->> (parse-config :partner p
                                         :base-config-name base-config-name
                                         :config-directory config-directory
                                         :types types)
                            c/keys-in
                            set
                            (clojure.set/difference reference-config))]))

             (filter #(not (empty? (second %))))

             (map #(hash-map :compared-against compare-partner
                             :partner (first %)
                             :missing-keys (last %)))

             vec)))))
