(ns simply.messaging
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.string :as string]
            [simply.cqrs :as cqrs]))


;;;; COMPANIES

(def companies-topic "companies")

(defn company-message [type payload]
  {:pre [(string? type) (not (string/blank? type)) (map? payload)]}
  {:type type :payload payload})


(defmethod cqrs/handle-action :send-company-messages [t msgs {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t msgs
   (fn []
     (cqrs/send-messages cqrs-system companies-topic msgs))))


(defn send-company-messages-action [msgs]
  (cqrs/action :send-company-messages msgs))


(defn send-company-message-action [type payload]
  (send-company-messages-action [(company-message type payload)]))


;;;; SMS

(defn sms
  "* type is a string defining the sms type
   * reference is a string reference to the entity"
  ([type number text reference]
   {:type type :number number :text text :reference reference})
  ([type number text]
   {:type type :number number :text text}))


(def sms-topic "sms")

(defmethod cqrs/handle-action :send-smses [t smses {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t smses
   (fn []
     (cqrs/send-messages cqrs-system sms-topic smses))))


(defn send-multiple-smses-action [& smses]
  (cqrs/action :send-smses smses))


(defn send-sms-action [sms]
  (send-multiple-smses-action sms))


;;;; EMAILS


(defn email
  "* type is a string defining the email type
   * to can be a string or vec of strings
   * from needs to be a mailjet from email address
   * body can be text or html. If html then html? should be true
   * reference is a string reference to the entity
   * attachements is an vec of email-attachement"
  ([type to from subject body]
   (email type to from subject body false nil []))
  ([type to from subject body html?]
   (email type to from subject body html? nil []))
  ([type to from subject body html? reference]
   (email type to from subject body html? reference []))
  ([type to from subject body html? reference attachments]
   (let [to (if (string? to) [to] to)]
     {:type type :reference reference
      :from from
      :to to :subject subject :body body :is-html html?
      :attachments attachments})))


(defn email-attachment
  [mime-type name byte-array]
  {:content (String. (b64/encode byte-array))
   :type    mime-type
   :name    name})


(def email-topic "email")

(defmethod cqrs/handle-action :send-emails [t emails {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t emails
   (fn []
     (cqrs/send-messages cqrs-system email-topic emails))))


(defn send-multiple-emails-action [& emails]
  (cqrs/action :send-emails emails))


(defn send-email-action [email]
  (send-multiple-emails-action email))


;;;; SLACK

(defn slack
  [token channel type text]
  {:token token :channel channel :type type :text text})


(def slack-topic "slack")

(defmethod cqrs/handle-action :send-slacks [t slacks {:keys [cqrs cqrs-system]}]
  (cqrs/chunk-events
   cqrs t slacks
   (fn []
     (cqrs/send-messages cqrs-system slack-topic slacks))))


(defn send-multiple-slacks-action [& slacks]
  (cqrs/action :send-slacks slacks))


(defn send-slack-action [slack]
  (send-multiple-slacks-action slack))


;; Zendesk integration

(defn zendesk-ticket [type subject body reference]
  {:subject subject
   :type type
   :body body
   :reference reference})


(def zendesk-topic "zendesk-integration")


(defmethod cqrs/handle-action :zendesk/create-tickets
  [action-type tickets {:keys [cqrs cqrs-system]}]

  (cqrs/chunk-events
   cqrs action-type tickets
   (fn []
     (let [messages (map (fn [ticket]
                           {:type "CREATE-TICKET"
                            :ticket ticket})
                         tickets)]
       (cqrs/send-messages cqrs-system
                           "zendesk-integration"
                           messages)))))

;; Email test code

(comment
  "Example Email"
  (cqrs/command-result
   [(cqrs/event :foo {})]
   [(send-email-action (email "jou ma se mail" "a@b.com" "this-should-be-a-mailjet@b.com" "subject" "text-body" false "some ref number"))])

  "Example attachment (how you get the byte array is up to you)"
  (email-attachment "application/pdf"
                    "name.pdf"
                    (.toByteArray some-stream-stream)))
