(ns simply.wording-config-parser
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.edn :as edn]
            [clojure.walk :refer [postwalk]]))


;;;; READERS

(defprotocol WordingReference
  (path [this]))


(defn- wording-reference [v]
  (reify WordingReference
    (path [this] v)))


(s/def ::reference-path
  (s/or ::keyword keyword?
        ::keyword-vec (s/coll-of keyword? :kind vector? :min-count 1)))


(defn reference-reader
  [v]
  (s/assert ::reference-path v)
  (let [[t v] (s/conform ::reference-path v)]
    (if (= ::keyword t)
      (wording-reference [v])
      (wording-reference v))))


(defn wording-readers []
  {'sw/ref reference-reader})


;;;; REFERENCES

(defn resolve-references [o]
  (postwalk
   (fn [v]
     (loop [v v]
       (if (satisfies? WordingReference v)
         (recur (get-in o (path v)))
         v)))
   o))

(comment
  (resolve-references {:a {:a1 (reference-reader :b)}
                       :b "b"
                       :c {:c1 (reference-reader [:d :d1])}
                       :d {:d1 "d1"}
                       :e (reference-reader :f)
                       :f (reference-reader :g)
                       :g "g"}))

;;;; READ EDN

(defn read-edn-file [file]
  (edn/read-string {:readers (wording-readers)} (slurp file)))


(defn throw-config-file-not-found [file-path
                                   config-type]
  (let [error-message (str "No such wording config file in ./resources directory: "
                           file-path " for wording type: `" config-type "`. "
                           "Please ensure that an edn file is present for each wording config type.")]
    (throw (Exception. error-message))))


(defn config-file-path [config-directory-name
                        partner
                        config-type]
  (str config-directory-name "/"
       partner "/" (name config-type) ".edn"))


(defn load-config
  ([config-directory-name
    partner
    types]

   (load-config config-directory-name
                partner
                types
                {:throw-exception-if-no-config-file? true}))

  ([config-directory-name
    partner
    types
    {:keys [throw-exception-if-no-config-file?] :as options}]

   (into {}
         (map (fn [config-type]

                (let [file-path           (config-file-path config-directory-name
                                                            partner
                                                            config-type)

                      file                (io/resource file-path)

                      config-file-exists? (not (nil? file))]

                  (if config-file-exists?
                    {config-type (read-edn-file file)}
                    (when throw-exception-if-no-config-file?
                      (throw-config-file-not-found file-path
                                                   config-type)))))
              types))))


(defn parse-config [& {:keys [partner types]
                       :or {types [:all]}}]

  (let [default-base-config      "base"

        default-config-directory "wording-config"

        base-config              (load-config default-config-directory
                                              default-base-config
                                              types
                                              {:throw-exception-if-no-config-file? false})

        partner-config           (load-config default-config-directory
                                              partner
                                              types
                                              {:throw-exception-if-no-config-file? false})

        effective-config         (merge-with merge
                                             base-config
                                             partner-config)]
    (resolve-references effective-config)))
