(ns simply.wording-config-parser
  (:require [clojure.java.io :as io]))


(defn read-edn-file [file]
  (read-string (slurp file)))


(defn throw-config-file-not-found [file-path
                                   config-type]
  (let [error-message (str "No such wording config file in ./resources directory: "
                           file-path " for wording type: `" config-type "`. "
                           "Please ensure that an edn file is present for each wording config type.")]
    (throw (Exception. error-message))))


(defn config-file-path [config-directory-name
                        partner
                        config-type]
  (str config-directory-name "/"
       partner "/" (name config-type) ".edn"))


(defn load-config
  ([config-directory-name
    partner
    types]

   (load-config config-directory-name
                partner
                types
                {:throw-exception-if-no-config-file? true}))

  ([config-directory-name
    partner
    types
    {:keys [throw-exception-if-no-config-file?] :as options}]

   (into {}
         (map (fn [config-type]

                (let [file-path           (config-file-path config-directory-name
                                                            partner
                                                            config-type)

                      file                (io/resource file-path)

                      config-file-exists? (not (nil? file))]

                  (if config-file-exists?
                    {config-type (read-edn-file file)}
                    (when throw-exception-if-no-config-file?
                      (throw-config-file-not-found file-path
                                                   config-type)))))
              types))))


(def default-base-config "simply")


(def default-config-directory "wording-config")


(defn parse-config [& {:keys [config-directory-name
                              partner
                              types]
                       :or {config-directory-name default-config-directory
                            types [:all]}}]

  (let [base-config       (load-config config-directory-name
                                       default-base-config
                                       types)

        partner-provided? (not (nil? partner))

        partner-config    (if partner-provided?
                            (load-config config-directory-name
                                         partner
                                         types
                                         {:throw-exception-if-no-config-file? false})
                            {})

        effective-config (merge-with merge
                                     base-config
                                     partner-config)]
    effective-config))
