(ns simply.wording-config
  (:require [clojure.string :as string]
            [cljstache.core :as mustache-parser]))

(def default-type :all)


(def wording-config (atom {}))

(defn current-wording []
  @wording-config)


(defn load-config! [& {:keys [config]}]
  (reset! wording-config config))


(defn throw-if-config-nil [value type key]
  (if (nil? value)
    (let [error-message (str "Cannot find wording configuration for type = " type
                             ", key = " key ". Please check your wording configuration files.")]
      (throw (Exception. error-message)))))


(defn get-configured-value [type key]
  (get-in @wording-config
          [type key]))


(defn has-key? [& {:keys [type key]
                   :or {type default-type}}]

  (not (nil?
        (get-configured-value type
                              key))))


(defn get-wording [& {:keys [type key apply-template-params? params]
                      :or {type default-type
                           apply-template-params? true
                           params {}}}]
  (let [value (get-configured-value type key)]

    (throw-if-config-nil value type key)

    (if apply-template-params?
      (if (coll? value)
        (map #(mustache-parser/render % params)
             value)
        (mustache-parser/render value
                                params))
      ;; Raw template without params applied
      value)))


(defn get-email [& {:keys [type key signature apply-template-params? params]
                    :or {type default-type
                         apply-template-params? true
                         params {}}}]

  (let [email (get-wording :type type
                           :key key
                           :apply-template-params? apply-template-params?
                           :params params)]

    (if (nil? signature)
      email
      ;; Include signature
      (let [signature           (get-wording
                                 :type (or (:type signature)
                                           type)
                                 :key (:key signature)
                                 :apply-template-params? apply-template-params?
                                 :params (or (:params signature)
                                             params))
            [subject body]      email
            body-with-signature (str body "\n\n"
                                     signature)]
        [subject body-with-signature]))))


(defn get-email-as-map [& options]
  (let [[subject body] (apply get-email options)]
    {:subject subject
     :body body}))
