(ns simply-ux.state-store
  (:require [reagent.core :as reagent]))


(def app-state (reagent/atom {}))


;; Allows for an external atom to be set as the storage.
(def storage (atom nil))


(defn state-storage []
  "Returns an atom used for state storage.
   If an external atom is set using the `set-storage` function it will be returned.
   Alternativly the default `app-state` will be returned."
  (let [storage-atom @storage]
    (if storage-atom
      storage-atom
      app-state)))


(defn set-storage [storage-atom]
  "Sets the state storage to the provided `storage-atom` and allows for
   using an external atom as storage (as opposed to the default internal `app-state`)"
  (reset! storage storage-atom))


(defn current-state []
  @(state-storage))


(defn scoped-state-key [key]
  (if (coll? key)
    key
    [key]))


(defn scoped-current-state [key]
  @(reagent/cursor (state-storage)
                   (scoped-state-key key)))


(defn reset-state []
  (reset! (state-storage) {}))


(defn get-action-type [action]
  (cond
    (string? action)
    action

    (map? action)
    (:type action)

    (vector? action)
    (first action)))


(defn get-action-params [action]
  (cond
    (string? action)
    {}

    (map? action)
    (:data action)

    (and (vector? action)
         (> (count action) 1))
    (last action)

    :else {}))


(defn get-reducer [name reducers-config]
  (let [scope            (:scope reducers-config)

        all-reducers     (if scope
                           (:reducers reducers-config)
                           reducers-config)

        base-reducer    (get all-reducers
                             name)


        reducer-function (if (and scope
                                  base-reducer)
                           (fn [params state]
                             (update-in state (scoped-state-key scope)
                                        (fn [scoped-state]
                                          (base-reducer params (or scoped-state
                                                                   {})))))

                           base-reducer)]
    reducer-function))


(defn reduce-state [current-state
                    functions]
  (reduce (fn [state reducer]
            (reducer state))
          current-state
          functions))


(defn apply-functions-to-state [current-state
                                scope
                                functions]
  (if scope
    (update-in current-state
               (scoped-state-key scope)
               (fn [scoped-state]
                 (reduce-state scoped-state
                               functions)))
    (reduce-state current-state
                  functions)))


(defn apply-functions
  [state functions reducer-config]
  (if functions
    (apply-functions-to-state state
                              (:scope reducer-config)
                              functions)
    state))


(defn always-apply-functions
  [state reducer-config]
  (apply-functions state
                   (or (:always-apply reducer-config)
                       (:apply-after reducer-config))
                   reducer-config))


(defn apply-before-functions
  [state reducer-config]
  (apply-functions state
                   (:apply-before reducer-config)
                   reducer-config))


(defn reducer-config-with-scope
  [reducer-config scope]
  (if scope
    (if (:scope reducer-config)
      (assoc reducer-config :scope scope)
      {:scope scope
       :reducers reducer-config})
    reducer-config))


(defn state-has-changed?
  [original-state new-state]
  (not (= original-state new-state)))


(defn throw-unknown-reducer-error [action-type]
  (let [error-message (str "No reducers specified for " action-type " action.")]
    (throw (js/Error. error-message ))))


(defn dispatch-action [action reducer-config scope]
  (let [action-type (get-action-type action)

        action-data (get-action-params action)

        reducer     (get-reducer action-type
                                 (reducer-config-with-scope reducer-config
                                                            scope))]
    (if reducer
      (let [original-state  (current-state)

            state           (apply-before-functions
                             original-state
                             reducer-config)

            latest-state    (reducer action-data state)

            effective-state (always-apply-functions
                             latest-state
                             reducer-config)]
        (if (state-has-changed? original-state
                                effective-state)
          (reset! (state-storage) effective-state)))
      (throw-unknown-reducer-error action-type))))


(defn dispatch [& {:keys [action actions reducers scope]
                   :or {actions [action]}}]
  (doseq [action actions]
    (dispatch-action action
                     reducers
                     scope)))


;; TODO - Remove this in favour of state scope
(defn dissoc-in [state full-key]
  (let [remove-from (vec (butlast full-key))
        remove-key (last full-key)]
    (update-in state
               remove-from
               dissoc
               remove-key)))

;; TODO - Remove this in favour of state scope
(defn merge-in [state full-key value]
  (update-in state
             full-key
             merge
             value))


(defn dispatcher
  "Returns a function that can be called with an action as the parameter which will be
   dispatched against the provided `state-reducers`"
   [state-reducers]

  (fn [action]
    (dispatch :reducers state-reducers
              :action action)))


(defn action-dispatcher
  "Returns a function that accepts the action data as the first parameter and wraps a
   dispatch call using the provided `action` and  `state-reducers-or-dispatcher` (which can be either a map of reducerers or a dispatcher that was
   previously created using the `dispatcher` function.

   Allows for defining functions in component namespaces that wrap an action being dispatched and provides more readable code."
  [action state-reducers-or-dispatcher]

  (let [state-reducers-provided? (coll? state-reducers-or-dispatcher)]

    (fn [action-data]
      (let [action-to-dispatch [action action-data]]

        (if state-reducers-provided?
          (let [state-reducers state-reducers-or-dispatcher]
            (dispatch
             :action action-to-dispatch
             :reducers state-reducers))
          (let [dispatcher state-reducers-or-dispatcher]
            (dispatcher action-to-dispatch)))))))
