(ns simply-ux.ui-app
  (:require [reagent.core :as reagent]
            [clojure.string :as string]
            [simply-ux.state-store :as state-store]
            [goog.dom.safe :as safe-dom]))


(defn dom-event->value [event]
  (-> event
      .-target
      .-value))


(defn on-press-enter [handler]
  (fn [event]
    (let [key-pressed (string/lower-case (.-key event))]
      (if (= key-pressed "enter")
        (handler)))))


(defn focus-on-element [id]
  (if-let [element (.getElementById js/document id)]
    (.focus element)))


(defn emebeded-html-content [content]
  [:span
   {:dangerouslySetInnerHTML #js{:__html content}}])


(defn scroll-to-top []
  (.scrollTo js/window 0 0))


(defn navigate-to [path]
  (let [single-page-app-url (str "#" path)]
    (set! js/window.location.href
          single-page-app-url)))

;; External link

;; https://google.github.io/closure-library/api/goog.dom.safe.html#setLocationHref

(defn navigate-to-external-url [url]
  (safe-dom/setLocationHref
   js/window.location url))


;; Alias for backwards compatability of navigate-to-web-url function
(def navigate-to-web-url navigate-to-external-url)


(defn with-state [render state-scope props]
  (fn []
    (let [state (if (not state-scope)
                  (state-store/current-state)
                  (state-store/scoped-current-state
                   state-scope))]
      (if (not (empty? props))
        (render state props)
        (render state)))))


(defn render-without-state-as-param [component]
  (if (map? component)
    (:render component)
    component))


(defn component-acts-as-a-container? [component]
  (or (and (map? component)
           (or (:act-as-container? component)
               (:do-not-react-to-state-changes? component)))
      false))


(defn ->component [component]
  (let [state-scope     (:state-scope component)

        props           (merge {}
                               (:default-props component)
                               (:props component))


        render-function (if (not (component-acts-as-a-container? component))
                          (if (map? component)
                            (with-state (:render component)
                              state-scope props)
                            (with-state component state-scope props))

                          ;; State is NOT derefed when acting as a container
                          ;; and will not re-render the container
                          (render-without-state-as-param component))

        life-cyle      (if (map? component)
                         (-> component
                             (dissoc :render)
                             (dissoc :props)
                             (dissoc :default-props))
                         {})]
    [(with-meta
       render-function
       life-cyle)]))


(defn ->child-component
  "Wraps ->component in a function which allows you to use def to define components"
  [component]
  (fn []
    (->component component)))


(defn render-component
  [& {:keys [component element-id]}]
  (let [dom-element         (.getElementById
                             js/document element-id)

        component-to-render (->component
                             component)]

    (if (not (nil? dom-element))
      (do
        (scroll-to-top)

        (reagent/render
         component-to-render
         dom-element)))))


(defn run-delayed [{:keys [seconds]} handler]
  (let [seconds->miliseconds (* 1000 seconds) ]
    (js/setTimeout handler
                   seconds->miliseconds)))


(defn this-page []
  (.-hash js/window.location))


(defn with-input-value [event function]
  (let [value (dom-event->value event)]
    (function value)))


(def define-component ->child-component)
