(ns simply-ux.http-requests
  (:require [cljs-http.client :as http]
            [cljs.core.async :as async]
            [simply-ux.cookies :as cookies])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn successful-request? [response]
  (= 200 (:status response)))


(defn successful-response? [response]
  (= 200 (:status response)))


(defn not-found-response? [response]
  (= 404 (:status response)))


(defn validation-error-response? [response]
  (or (= 400 (:status response))
      (= 403 (:status response))))

(defn not-authorized-error-response? [response]
  (= 401 (:status response)))


(defn forbidden-error-response? [response]
  (= 403 (:status response)))


(defn error-response? [response]
  (let [status (:status response)]
    (not
     (nil?
      (some #{status} [500 400 403 404])))))

(defn response->error-message [response]
  (or (get-in response [:body :message])
      (get-in response [:body :error])))


(defn get-api-token []
  (cookies/get "token"))

(defn redirect! [loc]
  (set! (.-location js/window) loc))

(defn redirect-to-logout! []
  (redirect! (str (.-origin (.-location js/window)) "/logout")))


(defn make-request [request handler]
  (go
    (let [response (async/<! request)]
      (if (not-authorized-error-response? response)
        (redirect-to-logout!)
        (handler response)))))


;; Makes an authenticated request by including an `x-token` HTTP header which is supported by the panoramix-2 API.
(defn authenticated-request
  ([method url]
   (authenticated-request method url {}))

  ([method url options]
   (let [token (get-api-token)]
     (method url
             (merge options
                    {:headers {"x-token" token}})))))
