(ns simply-ux.state-store
  (:require [reagent.core :as reagent]))


(def app-state (reagent/atom {}))


(defn throw-unknown-reducer-error [action-type]
  (let [error-message (str "No reducers specified for " action-type " action.")]
    (throw (js/Error. error-message ))))


(defn current-state []
  @app-state)


(defn reset-state []
  (reset! app-state {}))


(defn get-action-type [action]
  (cond
    (string? action)
    action

    (map? action)
    (:type action)

    (vector? action)
    (first action)))


(defn get-action-params [action]
  (cond
    (string? action)
    {}

    (map? action)
    (:data action)

    (vector? action)
    (last action)))


(defn scoped-state-key [key]
  (if (coll? key)
    key
    [key]))


(defn scoped-current-state [key]
  (get-in (current-state)
          (scoped-state-key key)))


(defn get-reducer [name reducers-config]
  (let [scope            (:scope reducers-config)

        all-reducers     (if scope
                           (:reducers reducers-config)
                           reducers-config)

        base-reducer    (get all-reducers
                             name)


        reducer-function (if (and scope
                                  base-reducer)
                           (fn [params state]
                             (update-in state (scoped-state-key scope)
                                        (fn [scoped-state]
                                          (base-reducer params (or scoped-state
                                                                   {})))))

                           base-reducer)]
    reducer-function))


(defn reduce-state [current-state
                    functions]
  (reduce (fn [state reducer]
            (reducer state))
          current-state
          functions))


(defn apply-functions-to-state [current-state
                                scope
                                functions]
  (if scope
    (update-in current-state
               (scoped-state-key scope)
               (fn [scoped-state]
                 (reduce-state scoped-state
                               functions)))
    (reduce-state current-state
                  functions)))

(defn always-apply-functions [current-state reducer-config]
  (if-let [always-apply (:always-apply reducer-config)]
    (apply-functions-to-state current-state
                              (:scope reducer-config)
                              always-apply)
    current-state))


(defn reducer-config-with-scope [reducer-config
                                 scope]
  (if scope
    (if (:scope reducer-config)
      (assoc reducer-config :scope scope )
      {:scope scope
       :reducers reducer-config})
    reducer-config))


(defn dispatch-action [action reducer-config scope]
  (let [action-type (get-action-type action)

        action-data (get-action-params action)

        reducer     (get-reducer action-type
                                 (reducer-config-with-scope reducer-config
                                                            scope))]
    (if reducer
      (let [latest-state    (reducer action-data (current-state))
            effective-state (always-apply-functions
                             latest-state
                             reducer-config)]
        (reset! app-state effective-state))
      (throw-unknown-reducer-error action-type))))


(defn dispatch [& {:keys [action actions reducers scope]
                   :or {actions [action]}}]
  (doseq [action actions]
    (dispatch-action action
                     reducers
                     scope)))


(defn dissoc-in [state full-key]
  (let [remove-from (vec (butlast full-key))
        remove-key (last full-key)]
    (update-in state
               remove-from
               dissoc
               remove-key)))


(defn merge-in [state full-key value]
  (update-in state
             full-key
             merge
             value))
