(ns simply-ux.http-api
  (:require [cljs-http.client :as http]
            [simply-ux.ui-app :as ui-app]
            [simply-ux.http-requests :as http-requests]))


(defn ->http-method
  [http-method]
  (case http-method
    :get http/get
    :post http/post))


(defn poll-until
  [{:keys [url
           http-method
           params
           polling-successful-when

           ;; Polling Settings

           polling-interval
           try-count
           max-tries

           ;; Callbacks

           on-poll
           on-polling-success
           on-polling-timeout
           on-polling-error]

    :or {http-method :get
         params {}

         polling-interval 2
         try-count 0
         max-tries 5

         on-poll (fn [])
         on-polling-timeout (fn [])
         on-polling-error (fn [_])}

    :as polling-params}]

  (on-poll)

  (ui-app/run-delayed
   {:seconds polling-interval}
   (fn []
     (if (= try-count max-tries)
       (on-polling-timeout)

       (http-requests/make-request
        (http-requests/authenticated-request
         (->http-method http-method)
         url
         {:edn-params params})
        (fn [response]
          (if (http-requests/successful-request? response)
            (let [body (:body response)]
              (if (true? (polling-successful-when body))
                (on-polling-success body)
                (poll-until
                 (assoc polling-params
                        :try-count (inc try-count)))))
            (on-polling-error response))))))))


(defn request
  [{:keys [url
           http-method
           params
           on-request-start
           on-successful-request
           on-request-error]
    :or {http-method :post
         params {}
         on-request-start (fn [])
         on-successful-request (fn [])
         on-request-error (fn [_])}}]

  (on-request-start)

  (http-requests/make-request
   (http-requests/authenticated-request
    (->http-method http-method)
    url
    {:edn-params params})
   (fn [response]
     (if (http-requests/successful-request?
          response)
       (on-successful-request response)
       (on-request-error response)))))


(defn request-then-poll
  [& {{:keys [on-successful-request]
       :or {on-successful-request (fn [])}
       :as initial-request} :initial-request

      polling-params :polling}]

  (let [success-handler (fn [response]
                          (on-successful-request response)
                          (poll-until polling-params))]
    (request (assoc initial-request
                    :on-successful-request
                    success-handler))))
