# Simply ux

Simply ux is an application state and component driven framework based on [Reagent](http://reagent-project.github.io) which provides a simple yet powerful way to build ClojureScript applications.

By providing a minimal API and basic concepts, Simply ux aims at facilitating rapid application development and live coding during development as well as a production grade application framework.

[![Clojars Project](https://img.shields.io/clojars/v/za.co.simply/simply-ux.svg)](https://clojars.org/za.co.simply/simply-ux)


# Basic component example

The following code snippet includes code & comments that outline a basic component:

```clojure
(ns simply-ux-example-app.core
  (:require [simply-ux.state-store :as state-store]
            [simply-ux.ui-app :as ui-app]))

;; Define the state scope used by both the state reducers
;; and the component.
;;
;; Can be a single key or a vector of keys representing a nested location in the state.
(def state-scope :contact-details)


;; State reducer configuration associates reducer functions with action types.
;; A reducer function's primary role is to return new application state when the action is dispatched.
;; Specifying the `:scope` key instructions the reducers to operate on that key in the state.
(def state-reducers {:scope state-scope
                     :reducers {"SET-CONTACT-DETAILS" (fn [contact-details _]
                                                        contact-details)

                                "SET-FULL-NAME" (fn [full-name state]
                                                  (assoc state :full-name full-name))

                                "SET-EMAIL-ADDRESS" (fn [email-address state]
                                                      (assoc state :email-address email-address))

                                "SET-MOBILE-NUMBER" (fn [mobile-number state]
                                                      (assoc state :mobile-number mobile-number))}})


;; Components are represented as maps containing at minimum a :render key which represents the render function.
;; Since the application state is the most important data in the application it is passed in as the first parameter of the
;; render function.
;;
;; Components can include a `:state-scope` key which will limits the scope of the state the component uses.
;;
;; Components will reactively render only the when the state located at the `:state-scope` key changes.
(def view-contact-details-component
  {:state-scope state-scope

   :render (fn [state]
             [:div
              [:h4 "Your contact details"]

              [:label "Full name:"]
              [:div (:full-name state)]

              [:label "Email address:"]
              [:div (:email-address state)]

              [:label "Mobile number:"]
              [:div (:mobile-number state)]])})


;; The only way to change the application state is to dispatch an action
;; against a reducer config(specified as the `reducers` key)
;;
;; Once the action is dispatched and the associated state reducer runs and
;; returns the new state any components referring to the state that changed will be updated.
(state-store/dispatch
 :reducers state-reducers
 :action ["SET-CONTACT-DETAILS"
          {:full-name "Guilio Karadanais"
           :email-address "guilio@simply.co.za"
           :mobile-number "0731231234"}])


;; Renders the component to the `app` element in the DOM.
(ui-app/render-component
 :component view-contact-details-component
 :element-id "app")
```


# Concepts

1. [Working with application state using state store](https://gitlab.com/SimplyFinServices/simply-ux/blob/master/docs/state-store.md)
2. ~Building and rendering components~ (to be documented)
3. ~Client side routing~ (to be documented)
4. ~HTTP requests~ (to be documented)
5. ~Project setup and builds~ (to be documented)


# Deploying simply-ux to Clojars

The simply-ux library is made available via [https://clojars.org](https://clojars.org). 

In order to deploy you'll need a valid [clojars.org](https://clojars.org) account as well as a deploy token. You can request these from the Simply tech team.


## Step 1

One line 1 of the [`project.clj`](https://gitlab.com/SimplyFinServices/simply-ux/-/blob/master/project.clj#L1) file set the version number which represents the latest version, for example:

```
(defproject za.co.simply/simply-ux "1.0.7"
  ...)
```


## Step 2

From the terminal run the following command from within the simply-ux directory to deploy to [clojars.org](https://clojars.org):

```
cd simply-ux

GPG_TTY=$(tty) LEIN_SNAPSHOTS_IN_RELEASE=true lein deploy clojars
```

