(ns simply-ux.state-store
  (:require [reagent.core :as reagent]))


(def app-state (reagent/atom {}))


(defn throw-unknown-reducer-error [action-type]
  (let [error-message (str "No reducers specified for " action-type " action.")]
    (throw (js/Error. error-message ))))


(defn current-state []
  @app-state)


(defn reset-state []
  (reset! app-state {}))


(defn get-action-type [action]
  (cond
    (string? action)
    action

    (map? action)
    (:type action)

    (vector? action)
    (first action)))


(defn get-action-params [action]
  (cond
    (string? action)
    {}

    (map? action)
    (:data action)

    (vector? action)
    (last action)))


(defn scoped-state-key [key]
  (if (coll? key)
    key
    [key]))


(defn scoped-current-state [key]
  (get-in (current-state)
          (scoped-state-key key)))


(defn get-reducer [name reducers-config]
  (let [scope            (:scope reducers-config)

        all-reducers     (if scope
                           (:reducers reducers-config)
                           reducers-config)

        base-reducer    (get all-reducers
                             name)


        reducer-function (if (and scope
                                  base-reducer)
                           (fn [params state]
                             (update-in state (scoped-state-key scope)
                                        (fn [scoped-state]
                                          (base-reducer params (or scoped-state
                                                                   {})))))

                           base-reducer)]
    reducer-function))


(defn dispatch-action [action reducers]
  (let [action-type (get-action-type action)

        action-data (get-action-params action)

        reducer     (get-reducer action-type
                                 reducers)]
    (if reducer
      (let [latest-state (reducer action-data (current-state))]
        (reset! app-state latest-state))
      (throw-unknown-reducer-error action-type))))


(defn dispatch [& {:keys [action actions reducers]
                   :or {actions [action]}}]
  (doseq [action actions]
    (dispatch-action action
                     reducers)))


(defn dissoc-in [state full-key]
  (let [remove-from (vec (butlast full-key))
        remove-key (last full-key)]
    (update-in state
               remove-from
               dissoc
               remove-key)))


(defn merge-in [state full-key value]
  (update-in state
             full-key
             merge
             value))
