(ns hiccup-components.core
  (:require [clojure.string :as string]))

;; -- Private/internal API --

(def global-components (atom {}))


;; Searches both `local-components` and global components
;; for a `element-name` key and returns the value (which is a function)
;; or nil if not found.
;;
;; If the `element-name` is found in `local-components` it will take preference
;; over global components.
(defn- get-component-function
  [element-name local-components]
  (get (merge @global-components
              (or local-components
                  {}))
       element-name))


;; Checks if the given `element-name` is a component by
;; searching `local-components` and global components.
(defn- is-component?
  [element-name local-components]
  (not
   (nil? (get-component-function
          element-name
          local-components))))


;; Checks if the `element` is a component function definition which is a vector with the first element
;; being a map which includes the `:hiccup.components/element-name` key and the rest of the items
;; in the vector being parameters.
;;
;; For example `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
(defn- component-function-definition? [element]
  (and (coll? element)
       (map? (first element))
       (contains? (first element)
                  :hiccup.components/element-name)))


;; Will return items after the first item of a component function definition which
;; represents the paramters of a component function.
;;
;; Given a component function definition of:
;;
;; `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
;;
;; `["one" "two" "three"]` will be returned as the component function parameters.
(defn- ->component-function-params
  [element] (rest element))


;; Will return the `:hiccup.components/element-name` key of the first item in a
;; component function definition which represents the element-name of the component.
;;
;; Given a component function definition of:
;;
;; `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
;;
;; `:unordered-list` will be returned as the element name.
(defn- ->component-element-name
  [element]
  (:hiccup.components/element-name
   (first element)))


(defn- extract-component
  [element local-components]
  (let [component-function (get-component-function
                            (->component-element-name element)
                            local-components)

        component-params   (->component-function-params
                            element)]

    [component-function component-params]))


;; Indicates if the given `item` can have meta data.
;; Used to determine if a `key` value should be set in the meta data.
(defn- can-have-meta-data? [item]
  #?(:clj (instance? clojure.lang.IObj item))
  #?(:cljs (satisfies? IWithMeta item)))


;; Adds metadata with a `key` value to all items in `elements` to support Reagent and avoid
;; 'Each child in a list should have a unique "key" prop' react warnings.
;; If there is an existing key in either the elements' attributes or metadata that
;; key will be used otherwise a zero based, incremental index will be used as the key.
(defn- apply-list-keys [elements]
  (map-indexed
   (fn [index item]
     (if (can-have-meta-data? item)
       (let [meta-source (:key (or (meta item) {}))

             key-source  (when (coll? item)
                           (:key (nth item 1 {})))

             item-key   (or meta-source
                            key-source
                            (str index))]
         (with-meta item
           {:key item-key}))
       item))
   elements))


;; Used in conjunction with `clojure.walk/post-walk` checking the `element` and executing
;; didfferent logic based on the type of element to achieve expansio of components.
(defn- process-component
  [element
   local-components
   apply-to-component-output]
  (cond
    ;; If the element is a keyword check if the keyword has been registred
    ;; as a component and if so, replace the keyword with a map that includes a
    ;; `:hiccup.components/element-name` key which will get processed later.
    (keyword? element)
    (if (is-component? element local-components)
      {:hiccup.components/element-name element}
      element)


    ;; Check if the element represents a component function and if so, extract
    ;; the component function & params and execute the function.
    ;; The output of the component function is then provided to the `apply-to-component-output`
    ;; function which will recursively expand nested components.
    (component-function-definition? element)
    (let [[component-function
           component-params]  (extract-component element
                                                 local-components)
          component-output    (apply component-function
                                     component-params)]
      (apply-to-component-output
       component-output
       local-components
       apply-to-component-output))

    ;; If the element is a `seq?` then call `apply-list-keys` which will add metadata
    ;; with a `key` to support Reagent and avoid 'Each child in a list should have a unique "key" prop'
    ;; react warnings.
    ;; If there is an existing key in either the elements attributes or metabase that
    ;; key will be used.
    (seq? element)
    (apply-list-keys element)

    ;; Otherwise return the element as is.
    :else element))


;; Uses clojure.walk/postwalk in combination with the `process-component` function to
;; walk the given `document` and expand any component definitions.
;;
;; Will use global component definitions as well as any component definitions in `local-components`
;;
;; The `apply-to-component-output` function (last argument) will be applied to component output and
;; allows for handling nested components.
(defn- apply-components
  [document
   local-components
   apply-to-component-output]

  (clojure.walk/postwalk
   #(process-component
     % local-components apply-to-component-output)
   document))


;; Recurses over `document` and handles nested components.
;;
;; Calls the `apply-components` function and passes in the same `apply-components` function
;; as the last argument which is applied to component output to handled
;; nested component structures.
(defn- ->hiccup-recursive
  [document local-components]

  (apply-components document
                    local-components
                    apply-components))


;; -- Public API (includes docstrings) --

(defn clear-components []
  "Clears all components that where registered using `->reg-component`"
  (reset! global-components {}))


(defn reg-component
  "Registers a component with the given `element-name`(recommend to use a fully qualified keyword) and `component-function` which should return hiccup.

  - `element-name` - The element name that describes the component which can be used in Hiccup e.g `::ordered-list`, `::unordered-list`. Recommended to use fully qualified keywords to help organise components.

  - `component-function` - A pure function that takes parameters and returns Hiccup data, for example:"
  [element-name component-function]
  (swap! global-components
         assoc element-name component-function)

  element-name)


(defn ->hiccup
  "Processes components in the provided `hiccup-data` and returns expanded Hiccup data. Aliased as `expand-components` and `with-components`.

  - `hiccup-data` - The hiccup data to process which includes references to component element names.

  - `component-config` (Optional) A map of component configuration. Components defined here will overwrite components registered with `->reg-component` for that function call only."
  ([document local-components]
   (->hiccup-recursive
    document
    local-components))

  ([document]
   (->hiccup document {})))


;; Experimental aliases for ->hiccup mainly for use with Reagent / Re-frame / CLJS

(def expand-components
  "Alias of ->hiccup for use with Reagent/Re-frame" ->hiccup)

(def with-components
  "Alias of ->hiccup for use with Reagent/Re-frame" ->hiccup)
