# Hiccup components

[![Clojars Project](https://img.shields.io/clojars/v/za.co.simply/hiccup-components.svg)](https://clojars.org/za.co.simply/hiccup-components)

![](https://www.theseekerbooks.com/detroit/fisher_ext1.jpg)

Hiccup components' primary goal is to allow <ins>Hiccup to be defined as pure data</ins> while <ins>leveraging functions as a way to abstract components</ins>.

Hiccup components allows you to define abstracted components as functions that return [Hiccup](https://github.com/weavejester/hiccup) and associate them with element names that can be referred to in Hiccup much like standard HTML elements (`:div`, `h1`, `p` etc), without requiring the developer to call these functions directly.

Hiccup components works for both [Clojure Hiccup](https://github.com/weavejester/hiccup) and ClojureScript through [Reagent's hiccup-like syntax](https://reagent-project.github.io) which is also used in [Re-frame](https://github.com/day8/re-frame).

Imagine you could express Hiccup as a mixture of standard HTML elements and high level components:

```clojure
[:div 
  [::page-header "Profile details"]
  [::page-sub-header "Review and change your profile details"]]

  [:p "This information will be displayed on your profile page."]

  [::contact-details {:full-name "Martian Arts"
                      :email-address "area51@eatstatic.net"
                      :mobile-number "+30 807 1982"}]

  [::profile-details {:username "@martian-arts"
                      :location "Athens, Greece" 
                      :member-since "22 January 2011"}]

  [::profle-details-footer]]
```

Below is a simple example of defining a component and refering to it in Hiccup by its element name:

```clojure
(ns hiccup-components.example
  (:require [hiccup-components.core :as hc]))

;; Recommended to use fully qualified keywords to help organise components
(hc/reg-component
 ::unordered-list
 (fn [items]
   [:ul
    (for [item items]
      [:li item])]))

(hc/->hiccup
  [:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   ;; Refer to the `unordered-list` component instead of calling the function
   [::unordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]])

;; Expands into the following =>

[:div
 [:h1 "Testing a unordered list"]
 [:p "This is the start of the list"]
 [:ul ([:li "One"] [:li "Two"] [:li "Three"])]
 [:p "This is the end of the list"]]

```

# Addressing a problem in Hiccup

As it stands both Clojure [Hiccup](https://github.com/weavejester/hiccup) as well as [Reagent's hiccup-like syntax](https://reagent-project.github.io) allow you to define functions that return Hiccup but require the developer to call them directly as part of defining the Hiccup data:

```clojure
(defn unordered-list [items]
  [:ul
   (for [item items]
     [:li item])]))
```

For Clojure [Hiccup](https://github.com/weavejester/hiccup):

```clojure
[:div
  [:h1 "My list"]
  (unordered-list ["One" "Two" "Three"])]
```

With [Reagent's hiccup-like data](https://reagent-project.github.io):

```clojure
[:div
  [:h1 "My list"]
  [unordered-list ["One" "Two" "Three"]]]

;; or  

[:div
  [:h1 "My list"]
  (unordered-list ["One" "Two" "Three"])]
```

Being required to call these functions directly has the following limitations:

- Forces a mixture of data & functions.
- The more abstractions you have in the form of functions, the more your Hiccup looks like a series of nested function calls instead of structured data.
- Prevents hiccup from being defined using purely data.

Ideally we should be able to refer to and compose these functions that return hiccup like any other HTML element:

```clojure
[:div
  [:h1 "My lists"]
  [::unordered-list ["One" "Two" "Three"]]]
  [::ordered-list ["One" "Two" "Three"]]
  [:h1 "Contact details"]
  [:p "Your contact details will never be displayed publicly"]
  [::contact-details {:name "Merv Pepler" :email-address "merv@eatstatic.uk"}]]
```

# Defining components

Components can be defined in two ways, either using `reg-component` or defining your components in a map and passing them in as the second parameter to the `->hiccup` function.

Components are pure functions that take parameters, return [hiccup](https://github.com/weavejester/hiccup) and have an associated element name that can be referenced in hiccup.

## Registering your components using `reg-component`

Lets say you want to define two components that abstract an unordered and ordered list:

```clojure
(ns feature-annouce.core
  (:require [hiccup-components.core :as hc]))

;; Recommended to use fully qualified keywords to help organise components
(hc/reg-component
 ::unordered-list
 (fn [items]
   [:ul
    (for [item items]
      [:li item])]))

;; Recommended to use fully qualified keywords to help organise components
(hc/reg-component
 ::ordered-list
 (fn [items]
   [:ol
    (for [item items]
      [:li item])]))      
```

We now have two components registered with the element names of `unordered-list` and `ordered-list` which can referred to in Hiccup as follows:

```clojure
[:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

  ;; Not we are using the fully qualified keyword as the element name
   [::unordered-list ["One" "Two" "Three"]]

   [::ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]]
```

The `->hiccup` function is used to process the components and expand the Hiccup:

```clojure
(hc/->hiccup
  [:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   [::unordered-list ["One" "Two" "Three"]]

   [::ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]])

;; Expands into the following =>

[:div
 [:h1 "Testing a unordered list"]
 [:p "This is the start of the list"]
 [:ul ([:li "One"] [:li "Two"] [:li "Three"])]
 [:ol ([:li "One"] [:li "Two"] [:li "Three"])]
 [:p "This is the end of the list"]]
```

## Configuring components as maps

Alternatively components can be configured as maps with the key being the element name and the value being the component function:

```clojure
(def my-components
  {::unordered-list
   (fn [items]
     [:ul
      (for [item items]
        [:li item])])

   ::ordered-list
   (fn [items]
     [:ol
      (for [item items]
        [:li item])])})
```

The component map can then be passed in as the second parameter of `->hiccup`

```clojure
(hc/->hiccup
  [:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   [::unordered-list ["One" "Two" "Three"]]

   [::ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]]
  my-components)

  ;; Expands into the following =>

[:div
 [:h1 "Testing a unordered list"]
 [:p "This is the start of the list"]
 [:ul ([:li "One"] [:li "Two"] [:li "Three"])]
 [:ol ([:li "One"] [:li "Two"] [:li "Three"])]
 [:p "This is the end of the list"]]  
```

# Public API

- `(clear-components)` - Clears all components that where registered using `->reg-component`

- `(->reg-component element-name component-function)` - Registers a component with the given `element-name`(_recommend to use a fully qualified keyword_) and `component-function` which should return hiccup.

  - `element-name` - The element name that describes the component which can be used in Hiccup e.g `::ordered-list`, `::unordered-list`. Recommended to use fully qualified keywords to help organise components.

  - `component-function` - A pure function that takes parameters and returns Hiccup data, for example:

    ```clojure
    (fn [items]
      [:ul
       (for [item items]
         [:li item])])
    ```

  Example of `->reg-component`:

  ```clojure
  (hc/reg-component
   ::unordered-list
   (fn [items]
     [:ul
      (for [item items]
        [:li item])]))
  ```

- `(->hiccup hiccup-data component-config)` : Processes components in the provided `hiccup-data` and returns expanded Hiccup data. Aliased as `expand-components` and `with-components`.

  - `hiccup-data` - The hiccup data to process which includes references to component element names.

  - `component-config` (Optional) A map of component configuration. Components defined here will overwrite components registered with `->reg-component` for that function call only.

  Examples of `->hiccup`:

  ```clojure
  (hc/->hiccup
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [::unordered-list ["One" "Two" "Three"]]

     [::ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]])

  ;; or

  (hc/expand-components
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [::unordered-list ["One" "Two" "Three"]]

     [::ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]]) 

   ;; or

  (hc/with-components
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [::unordered-list ["One" "Two" "Three"]]

     [::ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]])           
  ```  

  Example passing in component config (_components defined in a map_) as the second parameter to `->hiccup`:

  ```clojure
  (hc/->hiccup
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [::unordered-list ["One" "Two" "Three"]]

     [::ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]]
     ;; Component config as map
     {::unordered-list
      (fn [items]
        [:ul
         (for [item items]
           [:li item])])

      ::ordered-list
      (fn [items]
        [:ol
         (for [item items]
           [:li item])])})
  ```
