(ns hiccup-components.core
  (:require [clojure.string :as string]))

;; -- Private/internal API --

(def global-components (atom {}))


;; Searches both `local-components` and global components
;; for a `element-name` key and returns the value (which is a function)
;; or nil if not found.
;;
;; If the `element-name` is found in `local-components` it will take preference
;; over global components.
(defn- get-component-function
  [element-name local-components]
  (get (merge @global-components
              (or local-components
                  {}))
       element-name))


;; Checks if the given `element-name` is a component by
;; searching `local-components` and global components.
(defn- is-component?
  [element-name local-components]
  (not
   (nil? (get-component-function
          element-name
          local-components))))


;; Checks if the `element` is a component function definition which is a vector with the first element
;; being a map which includes the `:hiccup.components/element-name` key and the rest of the items
;; in the vector being parameters.
;;
;; For example `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
(defn- component-function-definition? [element]
  (and (coll? element)
       (map? (first element))
       (contains? (first element)
                  :hiccup.components/element-name)))


;; Will return items after the first item of a component function definition which
;; represents the paramters of a component function.
;;
;; Given a component function definition of:
;;
;; `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
;;
;; `["one" "two" "three"]` will be returned as the component function parameters.
(defn- ->component-function-params
  [element] (rest element))


;; Will return the `:hiccup.components/element-name` key of the first item in a
;; component function definition which represents the element-name of the component.
;;
;; Given a component function definition of:
;;
;; `[{:hiccup.components/element-name :unordered-list}} ["one" "two" "three"]]`
;;
;; `:unordered-list` will be returned as the element name.
(defn- ->component-element-name
  [element]
  (:hiccup.components/element-name
   (first element)))


(defn- extract-component
  [element local-components]
  (let [component-function (get-component-function
                            (->component-element-name element)
                            local-components)

        component-params   (->component-function-params
                            element)]

    [component-function component-params]))


;; Indicates if the given `item` can have meta data.
;; Used to determine if a `key` value should be set in the meta data.
(defn- can-have-meta-data? [item]
  #?(:clj (instance? clojure.lang.IObj item))
  #?(:cljs (satisfies? IWithMeta item)))


;; Adds metadata with a `key` value to all items in `elements` to support Reagent and avoid
;; 'Each child in a list should have a unique "key" prop' react warnings.
;; If there is an existing key in either the elements' attributes or metadata that
;; key will be used otherwise a zero based, incremental index will be used as the key.
(defn- apply-list-keys [elements]
  (map-indexed
   (fn [index item]
     (if (can-have-meta-data? item)
       (let [meta-source (:key (or (meta item) {}))

             key-source  (when (coll? item)
                           (:key (nth item 1 {})))

             item-key   (or meta-source
                            key-source
                            (str index))]
         (with-meta item
           {:key item-key}))
       item))
   elements))


;; Used in conjunction with `clojure.walk/post-walk` checking the `element` and executing
;; certain logic based on the type of element.
(defn- process-component-base
  [element
   local-components
   apply-to-component-output]
  (cond
    ;; If the element is a keyword check if the keyword has been registred
    ;; as a component and if so, replace the keyword with a map that includes a
    ;; `:hiccup.components/element-name` key which will get processed later.
    (keyword? element)
    (if (is-component? element local-components)
      {:hiccup.components/element-name element}
      element)


    ;; Check if the element represents a component function and if so, extract
    ;; the component function & params and execute the function.
    ;; The output of the component function is then provided to the `apply-to-component-output`
    ;; function which will recursively expand nested components.
    (component-function-definition? element)
    (let [[component-function
           component-params]  (extract-component element
                                                 local-components)
          component-output    (apply component-function
                                     component-params)]
      (apply-to-component-output
       component-output
       local-components
       apply-to-component-output))

    ;; If the element is a `seq?` then call `apply-list-keys` which will add metadata
    ;; with a `key` to support Reagent and avoid 'Each child in a list should have a unique "key" prop'
    ;; react warnings.
    ;; If there is an existing key in either the elements attributes or metabase that
    ;; key will be used.
    (seq? element)
    (apply-list-keys element)

    ;; Otherwise return the element as is.
    :else element))


(defn- apply-components
  [document
   local-components
   apply-to-component-output]

  (clojure.walk/postwalk
   #(process-component-base
     % local-components apply-to-component-output)
   document))


(defn- ->hiccup-recursive
  [document local-components]

  (apply-components document
                    local-components
                    apply-components))


;; -- Public API --

(defn clear-components []
  (reset! global-components {}))


(defn reg-hiccup-component
  [element-name component-function]
  (swap! global-components
         assoc element-name component-function)

  element-name)

;; Alias for reg-hiccup-component

(def reg-component reg-hiccup-component)

(defn ->hiccup
  ([document local-components]
   (->hiccup-recursive
    document
    local-components))

  ([document]
   (->hiccup document {})))


;; Experimental aliases for ->hiccup mainly for use with Reagent / CLJS

(def expand-components ->hiccup)
(def with-components ->hiccup)
