# Hiccup components

[![Clojars Project](https://img.shields.io/clojars/v/za.co.simply/hiccup-components.svg)](https://clojars.org/za.co.simply/hiccup-components)

![](https://www.theseekerbooks.com/detroit/fisher_ext1.jpg)

Hiccup components' primary goal is to allow <u>hiccup to be defined as pure data</u> while leveraging functions as a way to abstract components.

Hiccup components allows you to define abstracted components as functions that return [hiccup](https://github.com/weavejester/hiccup) and associate them with element names. These element names can be referenced much like built in HTML elements without requiring the developer to call these functions directly.

# Addressing a problem in Hiccup

As it stands both Clojure [Hiccup](https://github.com/weavejester/hiccup) as well as [Reagent's hiccup-like syntax](https://reagent-project.github.io) allow you to define functions that return Hiccup but require the developer to call them directly as part of the Hiccup data:

```clojure
(defn unordered-list [items]
  [:ul
   (for [item items]
     [:li item])]))
```

For Clojure [Hiccup](https://github.com/weavejester/hiccup):

```clojure
[:div
  [:h1 "My list"]
  (unordered-list ["One" "Two" "Three"])]
```

With [Reagent's hiccup-like data](https://reagent-project.github.io):

```clojure
[:div
  [:h1 "My list"]
  (unordered-list ["One" "Two" "Three"])]

;; or

[:div
  [:h1 "My list"]
  [unordered-list ["One" "Two" "Three"]]]
```

Being required to call these functions directly has the following limitations:

- Forces a mixture of data and functions
- The more abstractions you have in the form of functions, the more your Hiccup looks like a series of nested function calls
- Couples defining data and executing functions when the data is evaluated.
- Prevents hiccup from being defined using purely data

Ideally we should be able to refer to and compose these functions that return hiccup like any other HTML element:

```clojure
[:div
  [:h1 "My list"]
  [:unordered-list ["One" "Two" "Three"]]]
  [:ordered-list ["One" "Two" "Three"]]
  [:contact-details {:name "Merv Pepler" :email-address "merv@eatstatic.uk"}]]
```

# Defining components

Components can be defined in two ways, either using `reg-hiccup-component` or defining your components in a map and passing them in as the second parameter to the `->hiccup` function.

Components are pure functions that take parameters, return [hiccup](https://github.com/weavejester/hiccup) and have an associated element name that can be referenced in hiccup.

## Registering your components using `reg-hiccup-component`

Lets say you want to define two components that abstracts an unordered and ordered list:

```clojure
(ns feature-annouce.core
  (:require [hiccup-components.core :as hc]))

(hc/reg-hiccup-component
 :unordered-list
 (fn [items]
   [:ul
    (for [item items]
      [:li item])]))

(hc/reg-hiccup-component
 :ordered-list
 (fn [items]
   [:ol
    (for [item items]
      [:li item])]))      
```

We now have two components registered with the element names of `unordered-list` and `ordered-list` which can be used in Hiccup as follows:

```clojure
[:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   [:unordered-list ["One" "Two" "Three"]]

   [:ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]]
```

The `->hiccup` function is used to process the components and expand the Hiccup:

```clojure
(hc/->hiccup
  [:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   [:unordered-list ["One" "Two" "Three"]]

   [:ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]])

;; Expands into the following:

[:div
 [:h1 "Testing a unordered list"]
 [:p "This is the start of the list"]
 [:ul ([:li "One"] [:li "Two"] [:li "Three"])]
 [:ol ([:li "One"] [:li "Two"] [:li "Three"])]
 [:p "This is the end of the list"]]
```

## Configuring components as maps

Alternatively components can be configured as maps with the key being the element name and the value being the component function:

```clojure
(def my-components
  {:unordered-list
   (fn [items]
     [:ul
      (for [item items]
        [:li item])])

   :ordered-list
   (fn [items]
     [:ol
      (for [item items]
        [:li item])])})
```

The component map can then be passed in as the second parameter of `->hiccup`

```clojure
(hc/->hiccup
  [:div
   [:h1 "Testing a unordered list"]

   [:p "This is the start of the list"]

   [:unordered-list ["One" "Two" "Three"]]

   [:ordered-list ["One" "Two" "Three"]]

   [:p "This is the end of the list"]]
  my-components)

  ;; Expands into the following:

[:div
 [:h1 "Testing a unordered list"]
 [:p "This is the start of the list"]
 [:ul ([:li "One"] [:li "Two"] [:li "Three"])]
 [:ol ([:li "One"] [:li "Two"] [:li "Three"])]
 [:p "This is the end of the list"]]  
```

# API

- `(clear-components)` - Clears all components that where registered using `->reg-hiccup-component`

- `(->reg-hiccup-component element-name component-function)` - Registers a component with the given `element-name` and `component-function` which should return hiccup.

  - `element-name` - The element name that describes the component which can be used in Hiccup e.g `:ordered-list`, `:unordered-list`

  - `component-function` - A pure function that takes parameters, constructs and returns Hiccup data, for example:

    ```clojure
    (fn [items]
      [:ul
       (for [item items]
         [:li item])])
    ```

  Example of `->reg-hiccup-component`:

  ```clojure
  (hc/reg-hiccup-component
   :unordered-list
   (fn [items]
     [:ul
      (for [item items]
        [:li item])]))
  ```

- `(->hiccup hiccup-data component-config)` : Processes components in the provided `hiccup-data` and returns expanded Hiccup data.

  - `hiccup-data` - The hiccup data to process which includes references to component element names.

  - `component-config` (Optional) A map of component configuration. Components defined here will overwrite components registered with `->reg-hiccup-component` for that function call only.

  Examples of `->hiccup`:

  ```clojure
  (hc/->hiccup
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [:unordered-list ["One" "Two" "Three"]]

     [:ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]])
  ```  

  ```clojure
  (hc/->hiccup
    [:div
     [:h1 "Testing a unordered list"]

     [:p "This is the start of the list"]

     [:unordered-list ["One" "Two" "Three"]]

     [:ordered-list ["One" "Two" "Three"]]

     [:p "This is the end of the list"]]
     ;; Component config as map
     {:unordered-list
      (fn [items]
        [:ul
         (for [item items]
           [:li item])])

      :ordered-list
      (fn [items]
        [:ol
         (for [item items]
           [:li item])])})
  ```
