(ns hiccup-components.core)

;; -- Private/internal API --

(def global-components (atom {}))


(defn- get-component-function
  [component-type local-components]
  (get (merge @global-components
              (or local-components
                  {}))
       component-type))


(defn- is-component?
  [component-type local-components]
  (not
   (nil? (get-component-function
          component-type
          local-components))))


(defn- component-function? [element]
  (and (coll? element)
       (map? (first element))
       (contains? (first element)
                  :hiccup.components/id)))


(defn- ->component-function-params
  [element]
  (rest element))


(defn- ->component-type
  [element]
  (:hiccup.components/id (first element)))


(defn- extract-component
  [element local-components]
  (let [component-function (get-component-function
                            (->component-type element)
                            local-components)

        component-params   (->component-function-params
                            element)]

    [component-function component-params]))


(defn- process-component-base
  [element
   local-components
   apply-to-component-output]
  (cond
    (keyword? element)
    (if (is-component? element local-components)
      {:hiccup.components/id element}
      element)


    (component-function? element)
    (let [[component-function
           component-params]  (extract-component element
                                                 local-components)
          component-output    (apply component-function
                                     component-params)]
      (apply-to-component-output
       component-output
       local-components
       apply-to-component-output))

    ;; Otherwise return the element as is.
    :else element))


(defn- apply-components
  [document
   local-components
   apply-to-component-output]

  (clojure.walk/postwalk
   #(process-component-base
     % local-components apply-to-component-output)
   document))


(defn- ->hiccup-recursive
  [document local-components]

  (apply-components document
                    local-components
                    apply-components))


;; -- Public API --

(defn clear-components []
  (reset! global-components {}))


(defn reg-hiccup-component
  [component-type component-function]
  (swap! global-components
         assoc component-type component-function))


(defn ->hiccup
  ([document local-components]
   (->hiccup-recursive
    document
    local-components))

  ([document]
   (->hiccup document {})))
