(ns xd-table.core
  (:require [reagent.core :as r]))

(defn index-of [coll el-search]
  (->> (map-indexed vector coll)
       (filter (fn [[i el]] (= el-search el)))
       ffirst))

(defn selection-default [& {:keys [model choices on-change label-fn]}]
  [:select
    {:value model
     :on-change (comp on-change #(-> % .-target .-value keyword))}
    (for [c choices]
      ^{:key (hash c)}
      [:option
        {:on-click (partial on-change (:id c))}
        (:id c)])])

(defn radio-button-default [& {:keys [model value label on-change]}]
  [:span.radio-button
    {:on-click (partial on-change value)}
    [:input
      {:type "radio" :value (or value "") :read-only true
       :checked (if (= model value) "checked" false)}]
    [:span label]])

(defn -xd-table
  [{:keys [cell-fn dim-yz? dim-xz? dims
           selection-com radio-button-com]
    :or {selection-com selection-default
         radio-button-com radio-button-default}}]
  (r/with-let [dim-x-ratom  (r/atom 0)
               dim-y-ratom  (r/atom 1)
               dim-yz-ratom (r/atom (when dim-yz? 2))
               dim-xz-ratom (r/atom (when dim-xz? 3))
               radio-ratoms (into {}
                              (for [{id :id} dims]
                                [id (r/atom nil)]))]
    (let
      [{vals-x :vals
        dim-l-fn-x :dim-label-fn
        dim-val-l-fn-x :dim-val-label-fn
        :or {dim-l-fn-x :id dim-val-l-fn-x str}} (nth dims @dim-x-ratom)

       {vals-y :vals
        dim-l-fn-y :dim-label-fn
        dim-val-l-fn-y :dim-val-label-fn
        :or {dim-l-fn-y :id dim-val-l-fn-y str}} (nth dims @dim-y-ratom)

       {vals-yz :vals
        dim-l-fn-yz :dim-label-fn
        dim-val-l-fn-yz :dim-val-label-fn
        :or {dim-l-fn-yz :id dim-val-l-fn-yz str}} (when-let [i @dim-yz-ratom]
                                                     (nth dims i))
       dim-yz-active? (and dim-yz? @dim-yz-ratom)

       {vals-xz :vals
        dim-l-fn-xz :dim-label-fn
        dim-val-l-fn-xz :dim-val-label-fn
        :or {dim-l-fn-xz :id dim-val-l-fn-xz str}} (when-let [i @dim-xz-ratom]
                                                     (nth dims i))
       dim-xz-active? (and dim-xz? @dim-xz-ratom)

       extra-dims
         (->> (map-indexed vector dims)
              (filter
                (fn [[i _]]
                  (not (contains? #{@dim-x-ratom @dim-y-ratom
                                    @dim-yz-ratom @dim-xz-ratom} i))))
              (map val)
              doall)
       extra-dim-vals
         (->> extra-dims
              (map (fn [{id :id}] @(get radio-ratoms id)))
              (remove nil?)
              doall)

       dim-x-selector
         [selection-com
           :model     (:id (nth dims @dim-x-ratom))
           :choices   dims
           :on-change #(reset! dim-x-ratom (index-of (map :id dims) %))
           :label-fn  dim-l-fn-x]
       dim-y-selector
         [selection-com
           :model     (:id (nth dims @dim-y-ratom))
           :choices   dims
           :on-change #(reset! dim-y-ratom (index-of (map :id dims) %))
           :label-fn  dim-l-fn-y]
       dim-yz-selector
         [selection-com
           :model     (if @dim-yz-ratom
                        (:id (nth dims @dim-yz-ratom))
                        :none)
           :choices   (conj dims {:id :none})
           :on-change #(reset! dim-yz-ratom
                               (if (= :none %)
                                 nil
                                 (index-of (map :id dims) %)))
           :label-fn  dim-l-fn-yz]
       dim-xz-selector
         [selection-com
           :model     (if @dim-xz-ratom
                        (:id (nth dims @dim-xz-ratom))
                        :none)
           :choices   (conj dims {:id :none})
           :on-change #(reset! dim-xz-ratom
                               (if (= :none %)
                                 nil
                                 (index-of (map :id dims) %)))
           :label-fn  dim-l-fn-xz]]

      [:div.zc-xd-table
        [:div.extra-dims
          (doall
            (for [{:keys [id dim-val-label-fn vals]
                   :or {dim-val-label-fn str}} extra-dims]
            (let [r (get radio-ratoms id)
                  r-val @r]
              ^{:key (str "extra-dim-" id)}
              [:div.extra-dim
                [:span.name id]
                [:div.vals
                  [:div.val
                    [radio-button-com
                      :model r-val
                      :value nil
                      :label "none"
                      :on-change #(reset! r nil)]]
                  (for [v vals]
                    ^{:key (hash v)}
                    [:div.val
                      [radio-button-com
                        :model r-val
                        :value v
                        :label (dim-val-label-fn v)
                        :on-change #(reset! r v)]])]])))]
        [:table
          [:thead
            [:tr
              (if (or dim-yz-active? (not dim-yz?))
                [:td.empty {:col-span 2}]
                [:td.empty.header-yz {:col-span 2}
                  dim-yz-selector])
              (when dim-yz-active?
                [:td.header-yz
                  dim-yz-selector])
              [:td.header-x
                {:col-span (cond-> (count vals-x)
                                   dim-xz-active? (* (count vals-xz)))}
                dim-x-selector]]
            [:tr
              (cond (and dim-xz? (not dim-xz-active?))
                    [:td.empty.header-yz-col {:col-span 2}
                      dim-xz-selector]
                    (and dim-yz? (not dim-yz-active?))
                    [:td.empty.header-yz-col {:col-span 2}]
                    :else
                    [:td.empty {:col-span 2}])
              (when dim-yz-active? [:td.header-yz-col])
              (doall
                (for [val-x vals-x]
                  ^{:key (str "cell-" (hash val-x))}
                  [:td.header-x-col
                    {:col-span
                      (cond-> 1
                              dim-xz-active? (* (count vals-xz)))}
                    (dim-val-l-fn-x val-x)]))]
            (when dim-xz-active?
              [:tr
                ; (if (or @dim-xz-ratom (not dim-yz?))
                ;   [:td.empty {:col-span (-> vals-y count)}])
                [:td.header-xz dim-xz-selector]
                [:td.header-xz.col]
                (when dim-yz-active?
                  [:td.header-xz.col])
                (for [val-x  vals-x
                      val-xz vals-xz]
                  ^{:key (str "cell-" (hash val-x) - (hash val-xz))}
                  [:td.header-xz.col (dim-val-l-fn-xz val-xz)])])]

          [:tbody
            (doall
              (for [val-y vals-y
                    [yz-i val-yz]
                      (if @dim-yz-ratom
                        (map-indexed vector vals-yz)
                        [[0 nil]])
                    :when (or @dim-yz-ratom (= 0 yz-i))]
              ^{:key (str "row-" (hash [val-y val-yz extra-dim-vals]))}
              [:tr
                (when (and (= val-y (first vals-y))
                           (= 0 yz-i))
                  [:td.header-y
                    {:row-span
                       (-> (count vals-y)
                           (* (if @dim-yz-ratom (count vals-yz) 1)))}
                    dim-y-selector])
                (if (and @dim-yz-ratom (not= 0 yz-i))
                  nil
                  [:td.header-y-row
                    (merge
                      (when (= val-y (last vals-y)) {:class "last"})
                      (when @dim-yz-ratom {:row-span (-> vals-yz count)}))
                    (dim-val-l-fn-y val-y)])
                (when @dim-yz-ratom
                  [:td.header-yz-row
                    (dim-val-l-fn-yz val-yz)])
                (doall (for [val-x vals-x
                             [xz-i val-xz]
                               (if @dim-xz-ratom
                                 (map-indexed vector vals-xz)
                                 [[0 nil]])
                             :when (or @dim-xz-ratom (= 0 xz-i))]
                  ^{:key  (str "cell-" (hash [val-x val-y xz-i]))}
                  [:td
                    (cell-fn
                      (merge
                        {(:id (nth dims @dim-x-ratom)) val-x
                         (:id (nth dims @dim-y-ratom)) val-y}
                        (when (and @dim-yz-ratom dim-yz?)
                          {(:id (nth dims @dim-yz-ratom)) val-yz})
                        (when (and @dim-xz-ratom dim-xz?)
                          {(:id (nth dims @dim-xz-ratom)) val-xz})
                        (->> extra-dims
                             (filter (fn [{id :id}] @(get radio-ratoms id)))
                             (map (fn [{:keys [id]}]
                                    [id @(get radio-ratoms id)]))
                             (into {}))))]))]))]]])))

(defn xd-table [& {:as args}] [-xd-table args])
