(ns navtree.core
  (:require [reagent.core :as r]
            [reagent.ratom :refer [reaction]]
            [taoensso.timbre :as tb]
            [keybind.core :as kb]
            [clojure.zip :as z]
            [navtree.zipper :as zn]))

(def realized-ds? #(cond (nil? %) true
                         (satisfies? IPending %) (realized? %)
                         :else true))

(def realization  #(cond (delay? %) (-> % deref doall)
                         :else      (-> % doall)))

(defn -loader [hiccup data-list]
  (r/create-class
    {:reagent-render (fn [hiccup _] (or hiccup [:span.loading-indicator]))
     :component-did-mount
     (fn [_]
       (js/setTimeout
         #(doseq [{:keys [d-val d-ratom d-real?]} data-list]
            (realization d-val)
            (reset! d-real? true)
         10)))}))

(defn -navtree-node
  [{:as params
    :keys [selection-path on-selection-path whitelist
           id-fn label-fn children-fn content-fn content-ratom
           loading-indicator]}
   id-path ;; terminal-headed
   tree]
  (r/with-let [id      (id-fn tree)
               id-path (conj id-path id)

               cc (when children-fn (children-fn tree))
               d  (when content-fn  (content-fn id))

               children-realized? (r/atom (realized-ds? cc))
               content-realized?  (r/atom (realized-ds? d))]

    (let [sel? (= (last selection-path) id)
          ex?  (cond whitelist  (contains? whitelist id)
                     :else true)

          data-realized?
          (cond-> true ex?  (and @children-realized?)
                       sel? (and @content-realized?))

          loader-args
          (cond-> (list)
            ex?  (conj {:d-val cc :d-real? children-realized?})
            sel? (conj {:d-val d  :d-real? content-realized?}))]

      ^{:key (hash id)}
      [:li
        {:class (str (when sel? "selection ")
                     (if ex? "expanded" "collapsed"))}
        ;; header
        [:span.header {:on-click (partial on-selection-path (reverse id-path))}
          (label-fn tree)]

        ;; loading indicator
        (if data-realized?
          (when sel? (do (reset! content-ratom (realization d)) nil))
          [-loader loading-indicator loader-args])

        ;; children
        (when (and ex? @children-realized?)
          [:ul.children
            (doall
              (for [c (realization cc)]
                ^{:key (hash (id-fn c))}
                [-navtree-node params id-path c]))])])))

(def handle-opt-delay #(if (delay? %) (deref %) %)) ;; fix

(defn nav-path [p z-fn [tree id-fn children-fn]]
  (let [children-fn (comp handle-opt-delay children-fn)
        z-nav       (zn/path->z-nav tree id-fn children-fn p)
        p-next      (zn/z-nav->path (cond-> z-nav (seq p) z-fn)
                                    id-fn)]
    [z-nav p-next]))

(defn init-bind!
  [sp-ratom z-nav-args on-selection-path k-fn-pairs]
  (doseq [[k z-fn] k-fn-pairs]
    (kb/bind! k k
      (fn [& _]
        (let [[_ sel-path-next] (nav-path @sp-ratom z-fn z-nav-args)]
          (tb/info "next selection-path: " sel-path-next)
          (on-selection-path sel-path-next))))))

(defn unbind! [kk] (doseq [k kk] (kb/unbind! k k)))

(defn jump-to-content!
  [sp-ratom z-nav-args on-selection-path content-fn]
  (loop []
    (let [[_ id-fn _] z-nav-args
          [z-nav sp-next] (nav-path @sp-ratom z/next z-nav-args)
          content         (-> z-nav z/node id-fn content-fn handle-opt-delay)]
      (when-not content
        (do (tb/info "jumping to: " sp-next)
            (on-selection-path sp-next)
            (recur))))))

(defn -navtree
  [{:as params
    :keys [tree id-fn children-fn
           selection-path selection-path-ratom selection-path-init
           on-selection-path
           keynav? jump-to-content?
           whitelist whitelist?
           content-fn content-placeholder content-ratom]}]
  (r/with-let
    [;; selection-path (root-headed)
     sp-ratom  (or selection-path-ratom
                   (r/atom selection-path-init))
     sp-update #(reset! sp-ratom %)

     c-ratom   (or content-ratom
                   (r/atom (or content-placeholder "")))

     ;; whitelist
     wl-ratom (when whitelist? (r/atom #{}))
     wl-update-sp (fn [p] (reset! wl-ratom (set p)))
     _ (wl-update-sp @sp-ratom)

     ;; handler
     on-selection-path
     (or on-selection-path
         #(do (sp-update %)
              (when whitelist?
                (wl-update-sp %))))

     z-nav-args [tree id-fn children-fn]
     k-fn-pairs
     (partition 2
       ["j" z/next "k" z/prev "h" z/up "shift-j" z/right "shift-k" z/left])
     _
     (when keynav?
       (init-bind! sp-ratom z-nav-args on-selection-path k-fn-pairs))
     _
     (when jump-to-content?
       (jump-to-content! sp-ratom z-nav-args on-selection-path content-fn))]

    (let [sp (or selection-path @sp-ratom)
          wl (or whitelist
                 (and whitelist?
                      (or (and selection-path (set selection-path))
                          @wl-ratom)))]
      [:div.navtree
        [:ul.tree
          ^{:key (hash (id-fn tree))}
          [-navtree-node
            (-> params
                (assoc  :content-ratom c-ratom
                        :on-selection-path on-selection-path)
                (cond-> whitelist? (assoc :whitelist wl)
                        true       (assoc :selection-path sp))
                (dissoc :tree))
            (list)
            tree]]

        (when (and content-fn
                   (not content-ratom))
          @c-ratom)])
    (finally (when keynav? (unbind! (map first k-fn-pairs))))))

(defn navtree [& {:as params}] (-navtree params))
