(ns elastic-input.core
  (:require [reagent.core :as r]))

;; ----------------------------------------------------------------------------
;; helpers

(defn escape [txt]
  (clojure.string/escape txt
    {\< "&lt;"
     \> "&gt;"
     \& "&amp;"
     "\n" "<br>"}))

;; ------------------------------------
;; focus-ctrl

(defn focus-ctrl-factory
  "Create a handler for component-did-mount and component-did-update"
  [focus-tracker reset-scroll]
  (fn [this]  
    (let [n (r/dom-node this)
          _ (reset-scroll n)
          {:keys [curr prev]} @focus-tracker]
      (cond (and (not prev) curr)
            (do (swap! focus-tracker #(assoc % :prev true))
                (.focus n))
            (and prev (not curr))
            (do (swap! focus-tracker #(assoc % :prev false))
                (.blur n))
            :else true))))

;; ------------------------------------
;; phantom-ctrl

(defn phantom-ctrl-factory [inspect report display-style]
  (fn [this]
    (let [n (r/dom-node this)
          _ (-> n .-style .-display (set! display-style))
          h (inspect n)
          _ (-> n .-style .-display (set! "none"))]
     (report h))))

(defn phantom-factory [inspect report visible-style sentinel-str]
  (let [phantom-ctrl (phantom-ctrl-factory inspect report visible-style)]
    (fn [opts]
      (r/create-class
        {:reagent-render
         (fn [{:as opts :keys [text width height]}]
           [:div.phantom
             {:style {:width width :height height}
              :dangerouslySetInnerHTML
               {:__html (str (escape text) sentinel-str)}}])
         :component-did-update phantom-ctrl
         :component-did-mount  phantom-ctrl}))))



;; ----------------------------------------------------------------------------
;; components

(defn scroll-l [dom-node] (set! (.-scrollLeft dom-node) 0))
(defn scroll-t [dom-node] (set! (.-scrollTop  dom-node) 0))
(defn tgt-val  [e]        (-> e .-target .-value))

(defn input
  [opts]
  (r/create-class
    {:reagent-render
      (fn [{:as   opts
            :keys [input-opts
                   text placeholder width
                   on-text-change on-blur]
            :or   {on-blur #()}}]
        [:input
          (merge input-opts
            {:value       text
             :placeholder placeholder
             :style       {:width width}
             :on-change   (comp on-text-change tgt-val)
             :on-blur     (comp on-blur        tgt-val)})])
     :component-did-update
       (comp scroll-l r/dom-node)}))

(defn textarea
  [opts]
  (r/create-class
    {:reagent-render
      (fn [{:as   opts
            :keys [textarea-opts
                   text placeholder height width
                   on-text-change on-blur]
            :or   {width nil;"100%"
                   on-blur #()}}]
        [:textarea
          (merge textarea-opts
            {:value       text
             :placeholder placeholder
             :style       {:height height :width width}
             :on-change   (comp on-text-change tgt-val)
             :on-blur     (comp on-blur        tgt-val)})])
     :component-did-update
       (comp scroll-t r/dom-node)}))

;; ------------------------------------
;; foc

(defn input-foc
  [opts]
  (let [focus-tracker (r/atom {:curr false :prev false})
        focus-ctrl    (focus-ctrl-factory focus-tracker scroll-l)]
    (r/create-class
      {:reagent-render
         (fn [{:as opts :keys [focused?]}]
           (when (not= focused? (:curr @focus-tracker))
             (swap! focus-tracker #(update-in % [:curr] not)))
           [input
             (merge opts
               {:input-opts
                 {:class (if (:curr @focus-tracker) "active" "inactive")}})])
       :component-did-update focus-ctrl
       :component-did-mount  focus-ctrl})))

(defn textarea-foc
  [opts]
  (let [focus-tracker (r/atom {:prev false :curr false})
        focus-ctrl    (focus-ctrl-factory focus-tracker scroll-t)]
    (r/create-class
      {:reagent-render
         (fn [{:as opts :keys [focused?]}]
           (when (not= focused? (:curr @focus-tracker))
             (swap! focus-tracker #(update-in % [:curr] not)))
           [textarea
             (merge opts
               {:textarea-opts
                 {:class (if focused? "active" "inactive")}})])
       :component-did-update focus-ctrl
       :component-did-mount  focus-ctrl})))

;; ------------------------------------
;; auto-w/h

(def min-w   "Minimum width in case of no text"    10)
(def extra-w "Additional space at the end of text" 5)

(defn input-auto-w
  [opts]
  (r/with-let
    [width   (r/atom min-w)
     min-w   (get opts :min-w min-w)
     extra-w (get opts :extra-w extra-w)
     phantom (phantom-factory #(.-clientWidth %)
                              #(reset! width (if (< % min-w)
                                               (+ min-w extra-w)
                                               (+ % extra-w)))
                              "inline-block" "")
     buffer-ratom (when (contains? opts :buffer?) (r/atom (:text opts)))]
    (let [opts (if (:buffer? opts)
                 (merge opts {:on-text-change #(reset! buffer-ratom %)
                              :on-blur        #((:on-text-change opts) %)
                              :text           @buffer-ratom})
                 opts)
          opts (assoc opts :width @width)]
      [:div.elastic-input.input-auto-w (:container opts)
        (if (contains? opts :focused?)
          [input-foc opts]
          [input     opts])
        [phantom (select-keys opts [:text])]])))

(defn textarea-auto-h
  [opts]
  (r/with-let
    [height  (r/atom 25)
     phantom (phantom-factory #(.-clientHeight %)
                              #(reset! height %)
                              "block"
                              "<br class='lbr'>")
     buffer-ratom (when (contains? opts :buffer?) (r/atom (:text opts)))]
    (let [opts (if (:buffer? opts)
                 (merge opts {:on-text-change #(reset! buffer-ratom %)
                              :on-blur        #((:on-text-change opts) %)
                              :text           @buffer-ratom})
                 opts)
          opts (assoc opts :height @height)]
      [:div.elastic-input.textarea-auto-h (:container opts)
        (if (contains? opts :focused?)
          [textarea-foc opts]
          [textarea     opts])
        [phantom (select-keys opts [:text :width])]])))
