(ns yunjia.util.db
  (:require [clojure.java.jdbc :as j]
            [clojure.string :as string]
            [honeysql.helpers :as h :refer :all :exclude [update]]
            [honeysql.core :as sql]
            [taoensso.timbre :as timbre
             :refer (log trace debug info warn error fatal report
                         logf tracef debugf infof warnf errorf fatalf reportf
                         spy get-env log-env)]
            [taoensso.timbre.profiling :as profiling
             :refer (pspy pspy* profile defnp p p*)]))

(defn insert!
  "clojure.java.jdbc/insert!的包装函数，只支持一种参数传递形式，并且始终使用默认选项。
  并且为记录增加create_time/modify_time字段。"
  [db-spec table row-map]
  (let [time (quot (System/currentTimeMillis) 1000)
        row (assoc row-map
              :create_time time
              :modify_time time)]
    (j/insert! db-spec table row)))

(defn insert-multi!
  "clojure.java.jdbc/insert-multi!的包装函数，只支持一种参数传递形式，并且始终使用默认选项。
  并且为每一条记录增加create_time/modify_time字段。"
  [db-spec table row-map-seq]
  (let [time (quot (System/currentTimeMillis) 1000)
        rows (map #(assoc %
                    :create_time time
                    :modify_time time) row-map-seq)]
    (j/insert-multi! db-spec table rows)))

(defn update!
  "clojure.java.jdbc/update!的包装函数。
  并且会自动更新modify_time字段。"
  [db-spec table set-map where-clause & options]
  (let [time (quot (System/currentTimeMillis) 1000)
        map-time (assoc set-map :modify_time time)]
    (apply j/update! db-spec table map-time where-clause options)))

(defn delete!
  "clojure.java.jdbc/delete!的包装函数"
  [db-spec table where-clause & options]
  (apply j/delete! db-spec table where-clause options))

(defn execute!
  "clojure.java.jdbc/execute!的包装函数。只支持一种参数传递形式，并且始终使用默认选项。"
  [db-spec sql-vec]
  (debug (string/join ", " sql-vec))
  (j/execute! db-spec sql-vec))

(defn query!
  "clojure.java.jdbc/query的包装函数。只支持一种参数传递形式，并且始终使用默认选项。"
  [db-spec sql-vec]
  (debug (string/join ", " sql-vec))
  (j/query db-spec sql-vec))

(defn honeysql-page
  "为honeysql的sql结构添加分页语句。例如：
  (-> (select :*)
      (from :a)
      (honeysql-page 5 30))
  page和per_page可以是string或整数。"
  [sql-map
   page
   per_page]
  (let [page (cond
               (not page) 1
               (not (string? page)) page
               (not (string/blank? page)) (Integer/parseInt page)
               :else 1)
        page (if (> page 0)
               page
               1)
        per_page (cond
                   (not per_page) 10
                   (not (string? per_page)) per_page
                   (not (string/blank? per_page)) (Integer/parseInt per_page)
                   :else 10)
        per_page (if (and (> per_page 0) (< per_page 300))
                   per_page
                   10)
        begin (* per_page (dec page))]
    (-> sql-map
        (limit per_page)
        (offset begin))))


;;**************************************************** 过滤部分字段 ****************************************************
(defn filter-fields
  "create by yan 2018/01/23 16:15"
  [datas]
  (map #(dissoc % :enable :create_id :create_time :modify_id :modify_time :migrate_time :migrate_name :remark)
       datas))

;;**************************************************** 分页查询限制 ****************************************************
(defn limit-sql
  "create by yan 2018/01/23 09:49"
  [select-sql page per-page]
  (let [
        ;;每页数据量
        sql-limit (if (string? per-page)
                    (Integer/parseInt per-page)
                    (or per-page 10))

        ;;分页起始数据
        page (if (string? page)
               (Integer/parseInt page)
               (or page 10))

        page (if page
               (if (> page 0)
                 (- page 1)
                 0)
               0)
        sql-offset (* sql-limit page)]
    (if (and
          page
          per-page)
      (-> select-sql
          (limit sql-limit)
          (offset sql-offset))
      select-sql)))

(defn limit-sql-ex
  "create by yan 2018/01/23 09:49"
  [select-sql start length]
  (let [
        ;;每页数据量
        sql-limit (if (string? length)
                    (Integer/parseInt length)
                    (or length 10))

        ;;分页起始数据
        sql-offset (if (string? start)
                     (Integer/parseInt start)
                     (or start 10))]
    (if (and
          start
          length)
      (-> select-sql
          (limit sql-limit)
          (offset sql-offset))
      select-sql)))

;;**********************************************************************************************************************
;;                                                     单表操作
;;**********************************************************************************************************************

;;**************************************************** 单次插入一条数据 ************************************************
(defn insert
  "创建数据，单次插入一条数据
   db-spec 数据库连接
   table-name 数据表名
   table-data 数据map
   例如：
   (create db-spec :a {:id 0})
   create by yan 2018/01/24 11:19"
  [db-spec table-name table-data]
  (let [current-time (quot (System/currentTimeMillis) 1000)]
    (insert! db-spec
             (keyword table-name)
             (merge table-data {:create_time current-time
                                :modify_time current-time}))
    true))

;;**************************************************** 单次插入多条数据 ************************************************
(defn insert-multi
  "创建数据，单次插入多条数据
   db-spec 数据库连接
   table-name 数据表名
   table-data 数据map
   例如：
   (create db-spec :a {:id 0})
   create by yan 2018/01/24 11:26"
  [db-spec table-name table-datas]
  (let [current-time (quot (System/currentTimeMillis) 1000)]
    (insert-multi! db-spec
                   (keyword table-name)
                   (map #(merge %
                                {:create_time current-time
                                 :modify_time current-time})
                        table-datas))
    true))

;;**************************************************** 单次查询符合条件的所有数据 enable = 1 ***************************
(defn query-more-enable
  "根据条件查询记录列表。自动添加enable=1条件。
  where条件的写法与honeysql保持一致。
  例如：
  (query-more-enable db-spec :foo [:= :a 1] [:< :b 100])
  create by yan 2017/05/17 16:30"
  [db-spec table-name & [where-seq page per-page]]
  (let [query-seq (-> (select :*)
                      (from (keyword table-name))
                      (where (into [:and
                                    [:= :enable 1]]
                                   where-seq))
                      (order-by [:create_time :desc])
                      (limit-sql page per-page))]
    (->> (sql/format query-seq)
         (query! db-spec))))

;;**************************************************** 单次查询符合条件的所有数据 enable = 1 并过滤 ********************
(defn query-more-filter
  "create by yan 2018/01/23 16:18"
  [db-spec table-name & [where-seq page per-page]]
  (-> (query-more-enable db-spec table-name where-seq page per-page)
      (filter-fields)))

;;**************************************************** 单次查询符合条件的所有数据 enable = 0 ***************************
(defn query-more
  "根据条件查询记录列表。不添加任何附加条件。
  where条件的写法与honeysql保持一致。
  例如：
  (query-more db-spec :foo [:= :a 1] [:< :b 100])
  create by yan 2017/05/17 16:33"
  [db-spec table-name & where-seq]
  (let [select-sql (-> (select :*)
                       (from (keyword table-name)))

        select-sql (-> (if (seq where-seq)
                         (apply where select-sql where-seq)
                         select-sql)
                       (sql/format))]
    (query! db-spec select-sql)))

;;**************************************************** 单单次查询符合条件的单条数据 ************************************
(defn query-one-enable
  "根据条件查询一条记录。自动添加enable=1条件。
  where条件的写法与honeysql保持一致。
  例如：
  (query-one-enable db-spec :foo [:= :a 1] [:< :b 100])
  create by yan 2017/05/17 16:32"
  [db-spec table-name & where-seq]
  (first (query-more-enable db-spec table-name where-seq 1 1)))

;;**************************************************** 单单次查询符合条件的单条数据 并过滤 *****************************
(defn query-one-filter
  "create by yan 2018/01/23 16:23"
  [db-spec table-name & where-seq]
  (->> (apply query-one-enable db-spec table-name where-seq)
       (conj [])
       (filter-fields)
       (first)))

;;**************************************************** 单次统计符合条件的数据量 ****************************************
(defn count-enable
  "create by yan 2018/01/23 10:18"
  [db-spec table-name & where-seq]
  (let [query-seq (-> (select [:%count.id :count])
                      (from (keyword table-name))
                      (where (into [:and
                                    [:= :enable 1]]
                                   where-seq)))]
    (->> (sql/format query-seq)
         (query! db-spec)
         (first)
         (:count))))

;;**************************************************** 更新符合条件的数据 **********************************************
(defn update
  "单表操作，更新数据。
  sset-data 插入数据，{:table-key value}
  where条件的写法与honeysql保持一致。
  (update db-spec :foo {:a 1} [:= :id 0])
  create by yan 2018/01/24 11:51"
  [db-spec table-name sset-data & where-seq]
  (let [current-time (quot (System/currentTimeMillis) 1000)
        sset-data-ex (assoc sset-data :modify_time current-time)

        update-sql (-> (h/update (keyword table-name))
                       (sset sset-data-ex)
                       (where (apply conj [:and] where-seq))
                       (sql/format))]
    (if (not-empty where-seq)
      (do
        (execute! db-spec update-sql)
        true)
      false)))

;;**************************************************** 逻辑删除符合条件的数据 ******************************************
(defn logic-delete
  "create by yan 2018/01/24 14:59"
  [db-spec table-name & where-seq]
  (let [current-time (quot (System/currentTimeMillis) 1000)
        sset-data {:enable      0
                   :modify_time current-time}

        update-sql (-> (h/update (keyword table-name))
                       (sset sset-data)
                       (where (apply conj [:and] where-seq))
                       (sql/format))]
    (if (not-empty where-seq)
      (do
        (execute! db-spec update-sql)
        true)
      false)))

;;**********************************************************************************************************************
;;                                                  主表sql操作 begin
;;**********************************************************************************************************************

;;**************************************************** 单次查询符合条件的所有数据 **************************************
(defn foremost-more
  "根据条件查询记录列表。
  where条件的写法与honeysql保持一致。
  例如：
  (query-more-enable db-spec :foo [[:= :foremost.a 1] [:< :foremost.b 100]] 1 10)
  create by yan 2018/03/05 13:40"
  [db-spec table-name & [where-seq start length]]
  (let [query-seq (-> (select :foremost.*)
                      (from [(keyword table-name) :foremost])
                      (where where-seq)
                      (order-by [:foremost.create_time :desc])
                      (limit-sql-ex start length))]
    (->> (sql/format query-seq)
         (query! db-spec))))

;;**************************************************** 单次查询符合条件的所有数据 enable = 1 ***************************
(defn foremost-more-enable
  "根据条件查询记录列表。自动添加enable=1条件。
  where条件的写法与honeysql保持一致。
  例如：
  (query-more-enable db-spec :foo [[:= :foremost.a 1] [:< :foremost.b 100]] 1 10)
  create by yan 2018/03/05 13:50"
  [db-spec table-name & [where-seq start length]]
  (foremost-more db-spec table-name
                 (conj (or where-seq [:and])
                       [:= :foremost.enable 1])
                 start length))

;;**************************************************** 单次查询符合条件的一条数据 **************************************
(defn foremost-one
  "根据条件查询记录列表
  where条件的写法与honeysql保持一致。
  例如：
  (query-more-enable db-spec :foo [[:= :foremost.a 1] [:< :foremost.b 100]] 1 1)
  create by yan 2018/03/05 13:55"
  [db-spec table-name & [where-seq]]
  (first (foremost-more db-spec table-name where-seq 0 1)))

;;**************************************************** 单次查询符合条件的一条数据 enable=1  ****************************
(defn foremost-one-enable
  "根据条件查询记录列表
  where条件的写法与honeysql保持一致。
  例如：
  (query-more-enable db-spec :foo [[:= :foremost.a 1] [:< :foremost.b 100]] 1 1)
  create by yan 2018/03/05 13:57"
  [db-spec table-name & [where-seq]]
  (first (foremost-more db-spec table-name
                        (conj (or where-seq [:and])
                              [:= :foremost.enable 1])
                        0 1)))

;;**************************************************** 单次统计符合条件的数据量 ****************************************
(defn foremost-count
  "create by yan 2018/01/23 10:18"
  [db-spec table-name & [where-seq]]
  (let [query-seq (-> (select [:%count.foremost.id :count])
                      (from [(keyword table-name) :foremost])
                      (where where-seq))]
    (->> (sql/format query-seq)
         (query! db-spec)
         (first)
         (:count))))

;;**************************************************** 单次统计符合条件的数据量 enable=1 *******************************
(defn foremost-count-enable
  "create by yan 2018/01/23 10:18"
  [db-spec table-name & [where-seq]]
  (foremost-count db-spec table-name
                  (conj (or where-seq [:and])
                        [:= :foremost.enable 1])))