(ns yunjia.util.rest
  "Restful API相关工具。"
  (:require [ring.util.response :as resp]
            [cheshire.core :as json]
            [clj-http.client :as client]
            [honeysql.helpers :as h :refer :all :exclude [update]]
            [honeysql.core :as sql]
            [yunjia.util.db :as db]
            [yunjia.util.string :refer [split]]
            [clojure.edn :as edn]
            [clojure.walk :as walk]
            [clojure.string :refer [blank?]]
            [clojure.walk :refer [stringify-keys]]
            [clojure.core.match :refer [match]]))

(def content-type "application/json; charset=utf-8")

(defn json-response
  "将body转换为json字符串。"
  [response]
  (update-in response [:body] (fn [body]
                                (cond
                                  (nil? body) body
                                  (string? body) body
                                  :else (json/encode body)))))

(defn- query*-to-response
  "如果response的body是query*函数的输出，转换为符合接口的响应格式。"
  [response]
  (match response
         {:body {:rows        rows
                 :total-count count}} (-> response
                                          (assoc :body rows)
                                          (resp/header "TotalCount" count))
         {:body {:rows rows}} (assoc response :body rows)
         :else response))

(defn response
  "处理响应map：
  1、如果body是query+函数的输出，转换为符合接口的响应格式。
  2、增加content-type。
  3、如果(associative? body)为真，将body转换为json字符串。"
  [response]
  (let [body (:body response)
        ]
    (-> response
        query*-to-response
        (resp/content-type content-type)
        json-response)))

(defn- process-client-response
  "客户请求rest服务，对得到的响应进行处理，如json解码等。"
  [response]
  (clojure.core/update response :body
                       #(if % (json/decode % true))))

(defn client-get
  "clj-http.client/get的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (let [resp (client/get url
                         (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn client-get-e
  "抛出异常版本"
  [url & [req]]
  (let [response (client-get url req)]
    (if (= (:status response) 200)
      response
      (throw (ex-info (str "http get error: " url) response)))))

(defn client-delete
  "clj-http.client/delete的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (let [resp (client/delete url
                            (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn- post-put-patch
  "clj-http.client/put/post/patch的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [method url & [req]]
  (let [req (if (:body req)
              (clojure.core/update req :body json/generate-string)
              req)
        req (merge {:content-type content-type} req)
        resp (case method
               :post (client/post url req)
               :put (client/put url req)
               :patch (client/patch url req))]
    (process-client-response resp)))

(defn- post-put-patch-e
  "抛出异常版本"
  [method url & [req]]
  (let [response (post-put-patch method url req)]
    (if ((set [200 201]) (:status response))
      response
      (throw (ex-info (str "http request error: " url) response)))))

(defn client-post
  "clj-http.client/post的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch :post url req))

(defn client-put
  "clj-http.client/put的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch :put url req))

(defn client-patch
  "clj-http.client/patch的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch :patch url req))

(defn client-post-e
  "抛出异常版本
  clj-http.client/post的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch-e :post url req))

(defn client-put-e
  "抛出异常版本
  clj-http.client/put的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch-e :put url req))

(defn client-patch-e
  "抛出异常版本
  clj-http.client/patch的包装函数，调用方式和原函数一致。
  请求的处理：增加了content-type，对请求body进行json编码。
  响应的处理：对body进行json解码。"
  [url & [req]]
  (post-put-patch-e :patch url req))

(defn- build-page-sql-map
  [per_page page]
  (let [per_page (cond
                   (= per_page "") 10
                   (string? per_page) (Integer/parseInt per_page)
                   :else per_page)
        per_page (if (and (> per_page 0) (< per_page 200))
                   per_page
                   10)
        page (cond
               (= page "") 1
               (string? page) (Integer/parseInt page)
               :else page)
        page (if (> page 0)
               page
               1)
        begin (* per_page (dec page))]
    (-> (limit per_page)
        (offset begin))))

(defn- add-alias-to-field
  "为数据库字段添加别名，返回keyword格式。"
  [field-to-alias field]
  (if-let [alias (get field-to-alias (keyword field))]
    (keyword (str (name alias) "." (name field)))
    (keyword field)))

(defn- build-sort-sql-map
  [sort-string field-to-alias]
  (let [order-seq (for [sp (split sort-string #",")
                        :let [field (if (.startsWith sp "-")
                                      (.substring sp 1)
                                      sp)
                              key-field (add-alias-to-field field-to-alias field)
                              desc (.startsWith sp "-")]]
                    (if desc
                      [key-field :desc]
                      [key-field :asc]))]
    (if (seq order-seq)
      {:order-by order-seq}
      {})))

(defn- select-fields
  [fields rows]
  (let [keys (->> (split fields #",")
                  (map keyword))]
    (if (empty? keys)
      rows
      (map #(select-keys % keys) rows))))

(defn- check-filter-tree
  "校验filter表达式树是否有效。"
  [filter-tree]
  (if-let [s (seq filter-tree)]
    (case (first s)
      (:and :or) (and (next s)
                      (every? #(check-filter-tree %) (next s)))
      (:= :> :< :>= :<= :not= :like) (nth s 2 nil)
      :in (not (empty? (nth s 2 nil)))
      :between (and (nth s 2 nil)
                    (nth s 3 nil))
      false)))

(defn- make-field-to-alias
  [alias-map]
  (->> alias-map
       (mapcat (fn [[alias field-seq]]
                 (map #(vector % alias) field-seq)))
       (into {})))

(defn- add-alias-to-filter-tree
  "为字段名增加数据表别名，如果是like操作，额外增加%符号。"
  [filter-tree alias-map]
  (let [op-set #{:= :> :< :>= :<= :not= :like :in :between}
        field-to-alias (make-field-to-alias alias-map)]
    (walk/prewalk #(if (and
                         (sequential? %)
                         (op-set (first %)))
                    (let [[op field & vs] %
                          field-with-alias (add-alias-to-field field-to-alias field)]
                      (if (= op :like)
                        (into [op field-with-alias] (map (fn [v] (str "%" v "%")) vs))
                        (into [op field-with-alias] vs)))
                    %)
                  filter-tree)))

(defn- build-filter-sql-cond
  "解析filter参数，构建用于where的honeysql查询条件。"
  [filter-string options]
  (if-let [filter-tree (edn/read-string filter-string)]
    (let [alias-map (:alias options)]
      (if (check-filter-tree filter-tree)
        (add-alias-to-filter-tree filter-tree alias-map)
        (throw
          (ex-info "invalid filter" {:filter filter-string}))))))

(defn query-options-alias
  "构造查询选项map，增加别名配置。"
  ([alias-map] (query-options-alias {} alias-map))
  ([options alias-map] (assoc options :alias alias-map)))

(defn query-options-enable
  "构造查询选项map，增加enable配置。"
  ([enable?] (query-options-enable {} enable?))
  ([options enable?] (assoc options :enable enable?)))

(defn filter->field-to-conds
  "从查询条件字符串中，提取出map：查询字段 -> 该字段的查询条件列表。"
  [filter-string]
  (let [tree (edn/read-string filter-string)
        branch? (fn [node]
                  (and (sequential? node)
                       (#{:and :or} (first node))))]
    (if tree
      (->> tree
           (tree-seq branch? rest)
           (filter #(not (#{:and :or} (first %))))
           (group-by second))
      {})))

(defn query*
  "增强的数据库查询函数。
  解析请求参数，根据honeysql-map查询语句查询数据库，并增加下列功能：
  - 字段选择
  - 数据总数
  - 分页
  - 排序功能
  - 添加enable=1条件
  - filter查询条件: 使用honeysql的where条件格式。https://github.com/jkk/honeysql
  具体参数请参考http://dev.enchant.com/api/v1。
  函数参数:
  - options: 一个map，用来控制query的行为。下面是一个完整的例子。
          {:alias {:a [:age :address :name]   ;; 这几个字段属于别名为a的table
                   :b [:create_time :level]   ;; 这几个字段属于别名为b的table
                   :c [:enable]               ;; 这几个字段属于别名为c的table
                   }
           :enable false                      ;; 默认增加enable=1的查询条件，设置为false则不增加该条件
          }
  返回一个map：
  :rows 查询的结果集。
  :total-count 如果有数据总数的请求，则这个key对应数据总数。"
  [db-spec request honeysql-map & [options]]
  (let [opts (merge {:enable true} options)
        {:keys [fields count per_page page sort filter]} (get-in request [:parameters :query])
        sort (if sort sort "")
        fields (if fields fields "")
        field-to-alias (make-field-to-alias (:alias opts))
        filter-sql-cond (build-filter-sql-cond filter opts)
        enable (add-alias-to-field field-to-alias :enable)
        where-sql-cond (if (:enable opts)
                         (if filter-sql-cond
                           [:and [:= enable 1] filter-sql-cond]
                           [:= enable 1])
                         filter-sql-cond)
        sql-map-where (-> honeysql-map
                          (merge-where where-sql-cond))
        sql-map (-> sql-map-where
                    (merge
                      (if (and per_page page)
                        (build-page-sql-map per_page page)
                        {}))
                    (merge (build-sort-sql-map sort field-to-alias)))
        rows (->> sql-map
                  sql/format
                  (db/query! db-spec))
        count-sql-map (-> (select [:%count.* :total_count])
                          (from [sql-map-where :_table_for_count_]))
        ;(merge sql-map-where (select [:%count.* :total_count]))
        total-count (if (or (= count true) (= count "true") (= count :true))
                      (->> count-sql-map
                           sql/format
                           (db/query! db-spec)
                           first
                           :total_count))
        result {:rows (if (blank? fields)
                        rows
                        (select-fields fields rows))}]
    (if total-count
      (assoc result :total-count total-count)
      result)))

(defn- embed-resource-seq-1
  "为资源序列嵌入单个相关资源"
  [db-spec resource-seq embed-map-entry]
  (let [[embed-name [embed-table-name embed-table-field embed-value-fn]] embed-map-entry
        values (map embed-value-fn resource-seq)
        q-sql (-> (select :*)
                  (from embed-table-name)
                  (where [:and
                          [:= :enable 1]
                          [:in embed-table-field values]])
                  sql/format)
        rows (if (seq values)
               (db/query! db-spec q-sql)
               [])
        embed-to-rows (group-by embed-table-field rows)]
    (map #(assoc % embed-name (embed-to-rows (embed-value-fn %)))
         resource-seq)))

(defn embed-resource-seq
  "为资源序列嵌入相关资源。
  每个相关资源对应的在原资源中添加一个key，value是该类相关资源列表。"
  [db-spec resource-seq embed-map]
  (reduce #(embed-resource-seq-1 db-spec %1 %2) resource-seq embed-map))

(defn embed-resource-one
  "为单个资源嵌入相关资源。"
  [db-spec resource embed-map]
  (first (embed-resource-seq db-spec [resource] embed-map)))
