(ns yunjia.util.middleware
  (:require [ring.util.http-response :as hr]
            [clojure.string :as s]
            [clojure.walk :refer [keywordize-keys]]
            [taoensso.carmine :as car]
            [clojure.string :as str]
            [clojure.string :as clj-string]
            [ring.util.request :as clj-request]
            [clojure.data.json :as clj-json]
            [ring.util.io :as ring-io]
            [buddy.core.hash :as bd-md5]
            [buddy.core.codecs :as bd-codecs])
  (:import (java.util UUID)))

(def ^:const auth-session-key :auth-access-time)

(defn make-auth-session
  "构造认证成功的session。"
  ([] (make-auth-session {}))
  ([session] (assoc session auth-session-key (System/currentTimeMillis))))

(defn wrap-authentication
  "处理认证的Ring中间件，返回Ring的handler。
  放行已认证的请求，拦截未认证的请求。
  通过session实现。在session中增加:auth-access-time，记录最后一次已通过认证请求的访问时间。
  seconds: session过期的秒数。
  auth-fail-handler: 如未通过认证，由该处理器产生响应。若无此参数，仅返回403响应。
  "
  [handler seconds & [auth-fail-handler]]
  (fn [request]
    (let [old-session (:session request)
          access-time (auth-session-key old-session)
          current-time (System/currentTimeMillis)
          auth-ok (and access-time
                       (> (- current-time access-time) 0)
                       (< (- current-time access-time) (* seconds 1000)))
          fail-handler (if auth-fail-handler
                         auth-fail-handler
                         (fn [_] (hr/forbidden)))]
      (if auth-ok
        ; 验证通过，放行，并更新session
        (let [response (handler request)
              not-found? (or (nil? response)                ; 兼容compojure
                             (= (str (:status response))
                                "404"))
              update-session (fn []
                               (let [new-session
                                     (cond
                                       ;; handler没有更新session
                                       (not (contains? response :session)) (assoc
                                                                             old-session
                                                                             auth-session-key
                                                                             current-time)
                                       ;; handler更新了session
                                       (:session response) (assoc
                                                             (:session response)
                                                             auth-session-key
                                                             current-time)
                                       ;; handler删除了session
                                       :else nil)]
                                 (assoc response :session new-session)))]
          (if not-found?
            response
            (update-session)))
        ; 验证未通过，需要删除session
        (-> request
            (assoc :session nil)
            fail-handler
            (assoc :session nil))))))

;;************************************************ 对数据进行加密验证 **************************************************
(defn- md5-encrypt
  "create by yan 2017/03/13 11:10"
  [config params timesmap service]
  (let [params-json-str (-> (merge (sorted-map) params)
                            (clj-json/write-str :escape-unicode false))

        authentication (get-in config [:value :authentication])

        {:keys [secret_key]} (authentication (keyword service))]
    (-> (str timesmap params-json-str secret_key)
        (bd-md5/md5)
        (bd-codecs/bytes->hex))))

;;************************************************ 对内部请求进行验证 **************************************************
(defn wrap-auth-encrypt
  "处理认证的Ring中间件，返回Ring的handler。
   放行已认证的请求，拦截未认证的请求。
   create by yan 2017/03/13 11:09"
  [handler config & [auth-fail-handler]]
  (fn [request]
    (let [{:keys [character-encoding]} request

          ;;读取请求body
          body-string (clj-request/body-string request)

          body-params (if-not (clj-string/blank? (str body-string))
                        (->> body-string
                             (clj-json/read-str)
                             (map (fn [[key value]]
                                    (vector (keyword key) value)))
                             (into {}))
                        {})

          ;;验证参数
          {:keys [timestamp ciphertext service]} body-params

          new-body-params (dissoc body-params :timestamp :ciphertext :service)

          ;;加密得到的结果
          encrypt-result (md5-encrypt config new-body-params timestamp service)

          ;;将body放回request
          new-request (->> (clj-json/write-str new-body-params
                                               :escape-unicode character-encoding)
                           (ring-io/string-input-stream)
                           (assoc request :body))

          fail-handler (if auth-fail-handler
                         auth-fail-handler
                         (fn [_] (hr/forbidden)))]
      (cond
        (not= encrypt-result ciphertext) fail-handler
        :else (handler new-request)))))

(defn wrap-authentication-white-list
  "类似wrap-authentication的中间件，增加了路径白名单的功能。
  path-white-list-map:
      {:prefix-match []
       :full-match []}"
  [handler path-white-list-map config & [auth-fail-handler]]
  (let [{:keys [prefix-match full-match]} path-white-list-map
        full-match-set (->> full-match
                            (map #(if (clj-string/ends-with? % "/") % (str % "/")))
                            set)
        valid-session-seconds (get-in config [:value :vali_session_seconds])

        auth-handler (wrap-authentication handler valid-session-seconds auth-fail-handler)

        auth-encrypt-handler (wrap-auth-encrypt handler config auth-fail-handler)

        prefix-match-uri? (fn [uri] (some #(clj-string/starts-with? uri %) prefix-match))
        full-match-uri? (fn [uri] (full-match-set
                                    (if (clj-string/ends-with? uri "/")
                                      uri
                                      (str uri "/"))))]
    (fn [request]
      (let [{uri         :uri
             remote-addr :remote-addr} request]
        (cond
          ;;白名单访问
          (or (full-match-uri? uri)
              (prefix-match-uri? uri)) (handler request)
          ;;内部访问
          (= remote-addr "127.0.0.1") (auth-encrypt-handler request)
          ;;session访问
          :else (auth-handler request))))))

(defn wrap-path-set
  "如果请求的uri路径在path-set集合中，使用path-set-handler处理请求，否则使用handler处理。"
  [handler path-set path-set-handler]
  {:pre [(set? path-set) (fn? handler) (fn? path-set-handler)]}
  (fn [{:keys [uri] :as request}]
    (if (path-set uri)
      (path-set-handler request)
      (handler request))))

(defn wrap-path-prefix
  "如果请求的uri路径以path-prefix为前缀，使用path-prefix-handler处理请求，否则使用handler处理。"
  [handler path-prefix path-prefix-handler]
  {:pre [(string? path-prefix) (fn? handler) (fn? path-prefix-handler)]}
  (fn [{:keys [uri] :as request}]
    (if (str/starts-with? uri path-prefix)
      (path-prefix-handler request)
      (handler request))))

(def ^:private session-key-prefix "yj-session-key-")

(defn- get-session-key
  [request param-name]
  (param-name (keywordize-keys (:params request))))

(defn- session-request
  "根据旧的请求，生成新的请求。"
  [old-request redis-conn-opts param-name]
  (let [request-session-key (get-session-key old-request param-name)
        session (if (and request-session-key
                         (s/starts-with? request-session-key session-key-prefix))
                  (car/wcar redis-conn-opts (car/get request-session-key)))
        session-key (if session
                      request-session-key
                      (str session-key-prefix (UUID/randomUUID)))]
    (merge old-request {:session     (or session {})
                        :session/key session-key})))

(defn- session-response
  "根据response的:session来更新session。"
  [request response redis-conn-opts session-key param-name header-name]
  (if (contains? response :session)   ; 响应中包含:session才处理
    (if-let [session (:session response)] ; session不为nil，更新，否则删除
      (car/wcar redis-conn-opts (car/set session-key session))
      (car/wcar redis-conn-opts (car/del session-key))))
  (let [request-session-key (get-session-key request param-name)
        response (dissoc response :session)]
    (if (not= session-key request-session-key)
      (assoc-in response [:headers header-name] session-key)
      response)))

(defn wrap-session-redis
  "session中间件，使用定制的请求参数以及header。
  对请求的处理：从redis中读出当前session，添加到request的:session。
  对响应的处理，如果响应的:session存在且非nil，更新session。如果存在且nil，删除session。
  接受下列选项：
    :param-name 请求中用于传输session key的请求参数字段名，默认yunjia-session-key
    :header-name 响应中用于传输session key的响应header字段名，默认yunjia-session-key
    "
  ([handler redis-conn-opts] (wrap-session-redis handler redis-conn-opts {}))
  ([handler redis-conn-opts options]
   (let [options (merge {:param-name  :yunjia-session-key
                         :header-name "yunjia-session-key"} options)
         options (update options :param-name keyword)
         {:keys [param-name header-name]} options]
     (fn [request]
       (let [new-request (session-request request redis-conn-opts param-name)
             response (handler new-request)]
         (session-response new-request response redis-conn-opts
                           (:session/key new-request) param-name header-name))))))

