(ns yunjia.util.business.city
  "省市区相关的业务操作。"
  (:require [honeysql.helpers :as h :refer :all :exclude [update]]
            [honeysql.core :as sql]
            [yunjia.util.db :as db]
            [yunjia.util.business.config :as config]))

(defn area-seq-by-city
  [config-db-spec city-code]
  (let [h-sql (-> (select :*)
                  (from :area)
                  (where [:and
                          [:= :enable 1]
                          [:= :city_code city-code]]))]
    (db/query config-db-spec (sql/format h-sql))))

(defn province-code-to-name
  [config-db-spec province-code-seq]
  (config/code-to-name config-db-spec province-code-seq :province :code :name))

(defn city-code-to-name
  [config-db-spec city-code-seq]
  (config/code-to-name config-db-spec city-code-seq :city :code :name))

(defn area-code-to-name
  [config-db-spec area-code-seq]
  (config/code-to-name config-db-spec area-code-seq :area :code :name))

(defn city-code-by
  [^String area-code]
  (str (subs area-code 0 4) "00"))

(defn province-code-by
  [^String area-code-or-city-code]
  (str (subs area-code-or-city-code 0 2) "0000"))

(defn area-code->province-city-area
  "通过区县代码，获取省市区所有信息。
  返回map如下。
  key: area-code
  value: {:area_code area_code
          :city_code city_code
          :province_code province_code
          :area_name area_name
          :city_name city_name
          :province_name province_name}"
  [config-db-spec area-code-seq]
  (let [area-code-to-code-map (->> (for [area-code area-code-seq]
                                     [area-code {:area_code     area-code
                                                 :city_code     (city-code-by area-code)
                                                 :province_code (province-code-by area-code)}])
                                   (into {}))
        a-code-to-name (area-code-to-name config-db-spec area-code-seq)
        c-code-to-name (city-code-to-name
                         config-db-spec
                         (map :city_code (vals area-code-to-code-map)))
        p-code-to-name (province-code-to-name
                         config-db-spec
                         (map :province_code (vals area-code-to-code-map)))]
    (for [[area-code {:keys [area_code city_code province_code]}] area-code-to-code-map
          :let [area_name (a-code-to-name area_code)
                city_name (c-code-to-name city_code)
                province_name (p-code-to-name province_code)]]
      [area-code {:area_code area_code
                  :city_code city_code
                  :province_code province_code
                  :area_name area_name
                  :city_name city_name
                  :province_name province_name}])))



