(ns yunjia.util.time
  (:import (java.time ZoneId Instant LocalTime LocalDate ZonedDateTime LocalDateTime Period)
           (java.time.format DateTimeFormatter)))

(def zone-shanghai (ZoneId/of "Asia/Shanghai"))

(def formatter-yyyyMMddHHmm (DateTimeFormatter/ofPattern "yyyyMMddHHmm"))

(defn ^String now-yyyyMMddHHmm
  []
  (let [dt (ZonedDateTime/ofInstant (Instant/now) zone-shanghai)]
    (.format dt formatter-yyyyMMddHHmm)))

(defn calc-years
  "计算某个时间戳到当前时间的年数，不足整年部分也算一年。"
  [epoch-second]
  (let [instant (Instant/ofEpochSecond epoch-second)
        datetime (LocalDateTime/ofInstant instant zone-shanghai)
        date (.toLocalDate datetime)
        today (.toLocalDate (LocalDateTime/now zone-shanghai))
        period (Period/between date today)]
    (.getYears period)))

(defn yyyyMMdd->timestamp
  "日期格式为'yyyyMMdd‘转换成时间戳"
  [date-string]
  (let [formatter (DateTimeFormatter/ofPattern "yyyyMMdd")
        local-date (LocalDate/parse date-string formatter)
        local-time (LocalTime/of 0 0)]
    (-> (ZonedDateTime/of local-date local-time zone-shanghai)
        .toEpochSecond)))

(defn timestamp->datatime
  "时间戳解析成包括各单位的map"
  [timestamp]
  (let [instant (Instant/ofEpochSecond timestamp)
        zoned-data-time (ZonedDateTime/ofInstant instant zone-shanghai)]
    {:year (.getYear zoned-data-time)
     :month (.getMonthValue zoned-data-time)
     :day (.getDayOfMonth zoned-data-time)
     :hour (.getHour zoned-data-time)
     :minute (.getMinute zoned-data-time)
     :second (.getSecond zoned-data-time)}))
