(ns yunjia.util.middleware
  (:require [ring.util.http-response :as hr]
            [clojure.string :as s]
            [clojure.walk :refer [keywordize-keys]]
            [taoensso.carmine :as car])
  (:import (java.util UUID)))

(def ^:const auth-session-key :auth-access-time)

(defn make-auth-session
  "构造认证成功的session。"
  ([] (make-auth-session {}))
  ([session] (assoc session auth-session-key (System/currentTimeMillis))))

(defn wrap-authentication
  "处理认证的Ring中间件，返回Ring的handler。
  放行已认证的请求，拦截未认证的请求。
  通过session实现。在session中增加:auth-access-time，记录最后一次已通过认证请求的访问时间。
  seconds: session过期的秒数。
  auth-fail-handler: 如未通过认证，由该处理器产生响应。若无此参数，仅返回403响应。
  "
  [handler seconds & [auth-fail-handler]]
  (fn [request]
    (let [old-session (:session request)
          access-time (auth-session-key old-session)
          current-time (System/currentTimeMillis)
          auth-ok (and access-time
                       (> (- current-time access-time) 0)
                       (< (- current-time access-time) (* seconds 1000)))
          fail-handler (if auth-fail-handler
                         auth-fail-handler
                         (fn [_] (hr/forbidden)))]
      (if auth-ok
        ; 验证通过，放行，并更新session
        (let [response (handler request)
              not-found? (or (nil? response)                ; 兼容compojure
                             (= (str (:status response))
                                "404"))
              update-session (fn []
                               (let [new-session
                                     (cond
                                       ;; handler没有更新session
                                       (not (contains? response :session)) (assoc
                                                                             old-session
                                                                             auth-session-key
                                                                             current-time)
                                       ;; handler更新了session
                                       (:session response) (assoc
                                                             (:session response)
                                                             auth-session-key
                                                             current-time)
                                       ;; handler删除了session
                                       :else nil)]
                                 (assoc response :session new-session)))]
          (if not-found?
            response
            (update-session)))
        ; 验证未通过，需要删除session
        (-> request
            (assoc :session nil)
            fail-handler
            (assoc :session nil))))))

(defn wrap-authentication-white-list
  "类似wrap-authentication的中间件，增加了路径白名单的功能。
  path-white-list-map:
      {:prefix-match []
       :full-match []}"
  [handler path-white-list-map seconds & [auth-fail-handler]]
  (let [{:keys [prefix-match full-match]} path-white-list-map
        full-match-set (->> full-match
                            (map #(if (s/ends-with? % "/") % (str % "/")))
                            set)
        auth-handler (wrap-authentication handler seconds auth-fail-handler)
        prefix-match-uri? (fn [uri] (some #(s/starts-with? uri %) prefix-match))
        full-match-uri? (fn [uri] (full-match-set
                                    (if (s/ends-with? uri "/")
                                      uri
                                      (str uri "/"))))]
    (fn [request]
      (let [{uri :uri} request]
        (if (or (full-match-uri? uri)
                (prefix-match-uri? uri))
          (handler request)
          (auth-handler request))))))

(defn wrap-path-set
  "如果请求的uri路径在path-set集合中，使用path-set-handler处理请求，否则使用handler处理。"
  [handler path-set path-set-handler]
  {:pre [(set? path-set) (fn? handler) (fn? path-set-handler)]}
  (fn [{:keys [uri] :as request}]
    (if (path-set uri)
      (path-set-handler request)
      (handler request))))

(defn- get-session-key
  [request param-name]
  (param-name (keywordize-keys (:params request))))

(defn- session-request
  "根据旧的请求，生成新的请求。"
  [old-request redis-conn-opts param-name]
  (let [request-session-key (get-session-key old-request param-name)
        session (if (s/blank? request-session-key)
                  (car/wcar redis-conn-opts (car/get request-session-key)))
        session-key (if session
                      request-session-key
                      (str "yj-session-key-" (UUID/randomUUID)))]
    (merge old-request {:session     (or session {})
                        :session/key session-key})))

(defn- session-response
  "根据response的:session来更新session。"
  [request response redis-conn-opts session-key param-name header-name]
  (if (contains? response :session)   ; 响应中包含:session才处理
    (if-let [session (:session response)] ; session不为nil，更新，否则删除
      (car/wcar redis-conn-opts (car/set session-key session))
      (car/wcar redis-conn-opts (car/del session-key))))
  (let [request-session-key (get-session-key request param-name)
        response (dissoc response :session)]
    (if (not= session-key request-session-key)
      (assoc-in response [:headers header-name] session-key)
      response)))

(defn wrap-session-redis
  "session中间件，使用定制的请求参数以及header。
  对请求的处理：从redis中读出当前session，添加到request的:session。
  对响应的处理，如果响应的:session存在且非nil，更新session。如果存在且nil，删除session。
  接受下列选项：
    :param-name 请求中用于传输session key的请求参数字段名，默认yunjia-session-key
    :header-name 响应中用于传输session key的响应header字段名，默认yunjia-session-key
    "
  ([handler redis-conn-opts] (wrap-session-redis handler redis-conn-opts {}))
  ([handler redis-conn-opts options]
   (let [options (merge {:param-name  :yunjia-session-key
                         :header-name "yunjia-session-key"} options)
         options (update options :param-name keyword)
         {:keys [param-name header-name]} options]
     (fn [request]
       (let [new-request (session-request request redis-conn-opts param-name)
             response (handler new-request)]
         (session-response new-request response redis-conn-opts
                           (:session/key new-request) param-name header-name))))))


