(ns yunjia.util.db
  (:require [clojure.java.jdbc :as j]
            [clojure.string :as string]
            [honeysql.helpers :as h :refer :all :exclude [update]]
            [honeysql.core :as sql]
            [taoensso.timbre :as timbre
             :refer (log trace debug info warn error fatal report
                         logf tracef debugf infof warnf errorf fatalf reportf
                         spy get-env log-env)]
            [taoensso.timbre.profiling :as profiling
             :refer (pspy pspy* profile defnp p p*)]))

(defn insert!
  "clojure.java.jdbc/insert!的包装函数，只支持一种参数传递形式，并且始终使用默认选项。
  并且为记录增加create_time字段。"
  [db-spec table row-map]
  (let [time (quot (System/currentTimeMillis) 1000)
        row (assoc row-map :create_time time)]
    (j/insert! db-spec table row)))

(defn insert-multi!
  "clojure.java.jdbc/insert-multi!的包装函数，只支持一种参数传递形式，并且始终使用默认选项。
  并且为每一条记录增加create_time字段。"
  [db-spec table row-map-seq]
  (let [time (quot (System/currentTimeMillis) 1000)
        rows (map #(assoc % :create_time time) row-map-seq)]
    (j/insert-multi! db-spec table rows)))

(defn update!
  "clojure.java.jdbc/update!的包装函数"
  [db-spec table set-map where-clause & options]
  (let [time (quot (System/currentTimeMillis) 1000)
        map-time (assoc set-map :modify_time time)]
    (apply j/update! db-spec table map-time where-clause options)))

(defn delete!
  "clojure.java.jdbc/delete!的包装函数"
  [db-spec table where-clause & options]
  (apply j/delete! db-spec table where-clause options))

(defn execute!
  "clojure.java.jdbc/execute!的包装函数。只支持一种参数传递形式，并且始终使用默认选项。"
  [db-spec sql-vec]
  (debug (string/join ", " sql-vec))
  (j/execute! db-spec sql-vec))

(defn query
  "clojure.java.jdbc/query的包装函数。只支持一种参数传递形式，并且始终使用默认选项。"
  [db-spec sql-vec]
  (debug (string/join ", " sql-vec))
  (j/query db-spec sql-vec))

(defn honeysql-page
  "为honeysql的sql结构添加分页语句。例如：
  (-> (select :*)
      (from :a)
      (honeysql-page 5 30))
  page和per_page可以是string或整数。"
  [sql-map
   page
   per_page]
  (let [page (cond
               (not page) 1
               (not (string? page)) page
               (not (string/blank? page)) (Integer/parseInt page)
               :else 1)
        page (if (> page 0)
               page
               1)
        per_page (cond
                   (not per_page) 10
                   (not (string? per_page)) per_page
                   (not (string/blank? per_page)) (Integer/parseInt per_page)
                   :else 10)
        per_page (if (and (> per_page 0) (< per_page 300))
                   per_page
                   10)
        begin (* per_page (dec page))]
    (-> sql-map
        (limit per_page)
        (offset begin))))
