(ns yunjia.util.middleware
  (:require [ring.util.http-response :as hr]
            [clojure.string :as s]))

(def ^:const auth-session-key :auth-access-time)

(defn make-auth-session
  "构造认证成功的session。"
  []
  {auth-session-key (System/currentTimeMillis)})

(defn wrap-authentication
  "处理认证的Ring中间件，返回Ring的handler。
  放行已认证的请求，拦截未认证的请求。
  通过session实现。在session中增加:auth-access-time，记录最后一次已通过认证请求的访问时间。
  seconds: session过期的秒数。
  auth-fail-handler: 如未通过认证，由该处理器产生响应。若无此参数，仅返回403响应。
  "
  [handler seconds & [auth-fail-handler]]
  (fn [request]
    (let [access-time (get-in request [:session auth-session-key])
          current-time (System/currentTimeMillis)
          auth-ok (and access-time
                       (> (- current-time access-time) 0)
                       (< (- current-time access-time) (* seconds 1000)))
          fail-handler (if auth-fail-handler
                         auth-fail-handler
                         (fn [_] (hr/forbidden)))]
      (if auth-ok
        ; 验证通过，放行，并更新session
        (let [response (handler request)
              not-found (or (nil? response)                 ; 兼容compojure
                            (= (str (:status response))
                               "404"))]
          (if not-found
            response
            (update response :session assoc auth-session-key current-time)))
        ; 验证未通过，需要删除session
        (-> request
            (assoc :session nil)
            fail-handler
            (assoc :session nil))))))

(defn wrap-authentication-white-list
  "类似wrap-authentication的中间件，增加了路径白名单的功能。
  path-white-list-map:
      {:prefix-match []
       :full-match []}"
  [handler path-white-list-map seconds & [auth-fail-handler]]
  (let [{:keys [prefix-match full-match]} path-white-list-map
        full-match-set (->> full-match
                            (map #(if (s/ends-with? % "/") % (str % "/")))
                            set)
        auth-handler (wrap-authentication handler seconds auth-fail-handler)
        prefix-match-uri? (fn [uri] (some #(s/starts-with? uri %) prefix-match))
        full-match-uri? (fn [uri] (full-match-set
                                    (if (s/ends-with? uri "/")
                                      uri
                                      (str uri "/"))))]
    (fn [request]
      (let [{uri :uri} request]
        (if (or (full-match-uri? uri)
                (prefix-match-uri? uri))
          (handler request)
          (auth-handler request))))))




