(ns yunjia.util.rest
  "Restful API相关工具。"
  (:require [ring.util.response :as resp]
            [cheshire.core :as json]
            [clj-http.client :as client]
            [honeysql.helpers :refer [limit offset order-by select]]
            [honeysql.core :as sql]
            [yunjia.util.db :as db]
            [clojure.string :refer [split blank?]]
            [clojure.walk :refer [stringify-keys]]
            [clojure.core.match :refer [match]]))

(def content-type "application/json; charset=utf-8")

(defn json-response
  "将body转换为json字符串。"
  [response]
  (update-in response [:body] (fn [body]
                                (cond
                                  (nil? body) body
                                  (string? body) body
                                  :else (json/encode body)))))

(defn- query+-to-response
  "如果response的body是query+函数的输出，转换为符合接口的响应格式。"
  [response]
  (match response
         {:body {:rows        rows
                 :total-count count}} (-> response
                                          (assoc :body rows)
                                          (resp/header :Total-Count count))
         {:body {:rows rows}} (assoc response :body rows)
         :else response))

(defn response
  "处理响应map：
  1、如果body是query+函数的输出，转换为符合接口的响应格式。
  2、增加content-type。
  3、如果(associative? body)为真，将body转换为json字符串。"
  [response]
  (let [body (:body response)
        ]
    (-> response
        query+-to-response
        (resp/content-type content-type)
        json-response)))

(defn- process-client-response
  "客户请求rest服务，对得到的响应进行处理，如json解码等。"
  [response]
  (update response :body
          #(if % (json/decode % true))))

(defn client-get
  "clj-http.client/get的包装函数，调用方式和原函数一致。
  对请求做了一些处理，增加了content-type等。
  对响应的body进行json解码。"
  [url & [req]]
  (let [resp (client/get url
                         (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn client-post
  "clj-http.client/post的包装函数，调用方式和原函数一致。
  对请求做了一些处理，增加了content-type等。
  对响应的body进行json解码。"
  [url & [req]]
  (let [resp (client/post url
                          (merge {:content-type content-type} req))]
    (process-client-response resp)))

(defn- build-page-sql-map
  [per_page page]
  (let [per_page (cond
                   (= per_page "") 10
                   (string? per_page) (Integer/parseInt per_page)
                   :else per_page)
        per_page (if (and (> per_page 0) (< per_page 200))
                   per_page
                   10)
        page (cond
               (= page "") 1
               (string? page) (Integer/parseInt page)
               :else page)
        page (if (> page 0)
               page
               1)
        begin (* per_page (dec page))]
    (-> (limit per_page)
        (offset begin))))

(defn- build-sort-sql-map
  [sort-string sort-to-alias]
  (let [alias-map (stringify-keys sort-to-alias)
        order-seq (for [sp (split sort-string #",")
                        :when (not (.isEmpty sp))
                        :let [field (if (.startsWith sp "-")
                                      (.substring sp 1)
                                      sp)
                              alias (get alias-map field)
                              key-field (keyword (if alias
                                                   (str (name alias) "." field)
                                                   field))
                              desc (.startsWith sp "-")]]
                    (if desc
                      [key-field :desc]
                      key-field))]
    (if (seq order-seq)
      {:order-by order-seq}
      {})))

(defn- select-fields
  [fields rows]
  (let [keys (->> (split fields #",")
                  (filter (complement blank?))
                  (map keyword))]
    (if (empty? keys)
      rows
      (map #(select-keys % keys) rows))))

(defn query+
  "增强的数据库查询函数。
  解析请求参数，根据honeysql-map查询语句查询数据库，并增加字段选择、数据总数、分页、排序功能。
  具体参数请参考http://dev.enchant.com/api/v1。
  返回一个map：
  :rows 查询的结果集。
  :total-count 如果有数据总数的请求，则这个key对应数据总数。"
  [db-spec request honeysql-map & [sort-to-alias]]
  (let [{:keys [fields count per_page page sort]
         :or   {per_page 10 page 1 sort ""}} (get-in request [:parameters :query])
        sort (if sort sort "")
        fields (if fields fields "")
        per_page (if per_page per_page "")
        page (if page page "")
        sql-map (merge honeysql-map
                       (build-page-sql-map per_page page)
                       (build-sort-sql-map sort sort-to-alias))
        rows (->> sql-map
                  sql/format
                  (db/query db-spec))
        count-sql-map (merge honeysql-map (select [:%count.* :total_count]))
        total-count (if (or (= count true) (= count "true"))
                      (->> count-sql-map
                           sql/format
                           (db/query db-spec)
                           first
                           :total_count))
        result {:rows (if (blank? fields)
                        rows
                        (select-fields fields rows))}]
    (if total-count
      (assoc result :total-count total-count)
      result)))
