(ns yunjia.util.middleware)

(defn wrap-authentication
  "处理认证的Ring中间件，返回Ring的handler。
  放行已认证的请求，拦截未认证的请求。
  通过session实现。在session中增加:auth-access-time，记录最后一次已通过认证请求的访问时间。
  seconds: session过期的秒数。
  not-auth-handler: 如未通过认证，由该处理器产生响应。
  "
  [handler seconds not-auth-handler]
  (fn [request]
    (let [access-time (get-in request [:session :auth-access-time])
          current-time (System/currentTimeMillis)
          diff (- current-time access-time)
          auth-ok (and access-time
                       (> diff 0)
                       (< diff (* seconds 1000)))]
      (if auth-ok
        ; 验证通过，放行，并更新session
        (let [response (handler request)
              not-found (= (str (:status response))
                           "404")]
          (if not-found
            response
            (update response :session assoc :auth-access-time current-time)))
        ; 验证未通过，需要删除session
        (-> request
            (dissoc :session)
            not-auth-handler
            (dissoc :session))))))




