(ns yunjia.util.template.classic
  "hiccup风格的html模板库，经典风格。"
  (:require [hiccup.core :refer :all]
            [hiccup.page :refer :all]
            [medley.core :refer [map-vals]]))

(defn- fill-id-name
  "填充属性中的id和name。"
  [attributes]
  (let [v (or (:id attributes)
              (:name attributes))]
    (if v
      (merge attributes {:id v :name v})
      attributes)))

(defn make-head
  "创建hiccup格式head组件。
  option-map：选项map，接受如下的key：
    :title 可选。
  styles：样式文件列表。"
  [& [option-map styles]]
  [:head
   [:title (or (:title (map-vals str option-map))
               "")]
   [:meta {:charset "utf-8"}]
   [:meta {:http-equiv "X-UA-Compatible" :content "IE=edge"}]
   [:meta {:name "keywords" :content "index"}]
   [:meta {:name "viewport" :content "width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no"}]
   [:meta {:name "renderer" :content "webkit"}]
   [:meta {:http-equiv "Cache-Control" :content "no-siteapp"}]
   (apply include-css
          "/common/assets/css/amazeui.css"
          "/common/assets/css/public.css"
          "/common/assets/sweetalert/sweetalert.css"
          styles)])

(defn make-footer
  "创建hiccup格式footer组件。
  styles：javascript文件列表。"
  [& scripts]
  [:div
   (apply include-js
          "/common/assets/js/jquery.min.js"
          "/common/assets/js/amazeui.min.js"
          "/common/assets/js/jquery.json.min.js"
          "/common/assets/sweetalert/sweetalert.min.js"
          "/common/assets/js/public.js"
          scripts)])

(defn make-container
  "创建hiccup格式div空白容器。
  option-map: 选项map，接受如下的key:
    :class am-container/am-g 默认am-container。
    其他可用的html属性。"
  [option-map element & elements]
  (let [options (map-vals str (fill-id-name option-map))
        default {:class "am-container"}
        attributes (merge default options)]
    [:div attributes element elements]))

(defn- make-menu-1
  "创建一个两级菜单项"
  [root-id index name sub-nodes]
  (let [target-id (format "do-not-say-%d" index)
        data-am-collapse (format "{parent: '#%s', target: '#%s'}" root-id target-id)]
    [:div {:class "am-panel am-panel-default"}
     [:div {:class "am-panel-hd"}
      [:h4 {:class "am-panel-title" :data-am-collapse data-am-collapse}
       [:span {:class "am-icon-subway am-icon-sm"}]
       name]]
     [:div {:id target-id :class "am-panel-collapse am-collapse am-in"}
      [:div {:class "am-panel-bd"}
       [:ul {:class "am-list am-list-static am-list-border"}
        (for [node sub-nodes
              :let [onclick (format "openMenu('%s');" (:path node))
                    sub-name (:name node)]]
          [:li {:onclick onclick} [:span {:class "am-icon-home am-icon-sm"}] sub-name])]]]]))

(defn make-2-level-menu
  "创建hiccup格式两级菜单。"
  [tree name-key children-key & [id]]
  (let [root-id (or id "menu-root-default-id")
        attributes {:class "am-panel-group"
                    :id    root-id}
        tree-indexed (map-indexed vector tree)]
    [:div attributes
     (for [[i node] tree-indexed]
       (make-menu-1 root-id i (name-key node) (children-key node)))]))

(defn make-input
  "创建hiccup格式input组件。
  option-map: 选项map，接受如下的key:
    参考html的元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        default {:type        "text"
                 :class       "am-form-field"
                 :value       ""
                 :placeholder ""}
        attributes (merge default options)]
    [:input attributes]))

(defn make-textarea
  "创建hiccup格式textarea组件。
  option-map: 选项map，接受如下的key:
    参考html的元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        default {:rows  "6"
                 :cols  "20"
                 :value ""}
        attributes (merge default options)]
    [:textarea (dissoc attributes :value) (:value attributes)]))

(defn make-datepicker
  "创建hiccup格式datepicker组件。
  option-map: 选项map，接受如下的key:
    :format 默认yyyy-mm-dd。
    :setValue 默认2015-01-12。
    其他html元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        o-format (or (:format options)
                     "yyyy-mm-dd")
        o-setValue (or (:setValue options)
                       "2015-01-12")
        data-am-datepicker (format "{format: '%s', setValue: '%s'}" o-format o-setValue)
        default {}
        attributes (-> default
                       (merge options)
                       (dissoc :format :setValue))]
    [:input (merge attributes
                   {:type               "text"
                    :data-am-datepicker data-am-datepicker
                    :readonly           "readonly"})]))

(defn make-select
  "创建hiccup格式下拉菜单组件。
  option-seq: 下拉项序列，每个元素是一个下拉项map，支持的key:
    :content 下拉项内容。
    option元素支持的其他属性。
  option-map"
  [option-map option-element & option-elements]
  (let [options (map-vals str (fill-id-name option-map))
        default {:data-am-selected "{btnSize:'md'}"}
        elements (into [option-element] option-elements)]
    [:select (merge default options)
     (for [e elements]
       [:option (dissoc e :content) (:content e)])]))

(defn make-left-label-box
  "创建hiccup格式box组件，左边是文本，右边是输入框之类。
  right: 放在右边的组件，可以是make-input/make-textarea/make-datepicker，等等。
  txt: 放在左边的标签文本。
  txt-width: 文本宽度，默认为3。
  right-width: 右边组件的宽度，默认为9。"
  [right & [txt txt-width right-width]]
  [:div {:class "am-g am-margin-top"}
   [:div {:class (str "am-u-sm-" (or txt-width "3") " am-text-right")} (or txt "")]
   [:div {:class (str "am-u-sm-" (or right-width "9") " am-u-end col-end")} right]])

(defn make-form
  "创建hiccup格式form组件。
  option-map: 选项map，接受如下的key:
     :form-type 可选。am-container: 带有padding样式，container: 全屏样式。默认为container。
     其他html元素属性。
  elements: 表单元素组件列表。"
  [option-map element & elements]
  (let [options (map-vals str (fill-id-name option-map))
        form-type (or (:form-type options)
                      "container")
        default {:class  "am-form"
                 :method "POST"}
        attributes (-> default
                       (merge options)
                       (dissoc :form-type))]
    [:div {:class form-type}
     (into [:form attributes element] elements)]))

(defn make-panel
  "创建hiccup格式面板组件。
  option-map: 选项map，接受如下的key:
    :title
    :icon
    :target-id
    :color 例如am-panel-secondary，默认am-panel-default。"
  [option-map panel-content & [panel-footer]]
  (let [default {:color     "am-panel-default"
                 :title     ""
                 :target-id ""}
        options (merge default (map-vals str (fill-id-name option-map)))]
    [:div {:class (str "am-panel " (:color options))}
     [:div {:class "am-panel-hd"}
      (:title options)
      (if (:icon options)
        [:span {:data-am-offcanvas (format "{target:'#%s'}" (:target-id options))
                :class             (str "am-icon-sm fr " (:icon options))}])]
     [:div {:class "am-panel-bd"} panel-content]
     (if panel-footer
       [:div {:class "am-panel-footer"}
        [:div {:class "am-sm-12 nopd"}]
        panel-footer])]))

(defn make-list-op
  "创建hiccup格式下拉组件，支持点击操作，通常用于数据表格右边的操作栏。
  operation是一个map，表示一个下拉操作项，支持如下的key:
    :data-id
    :onclick
    :content"
  [operation & operations]
  [:div {:class "am-dropdown" :data-am-dropdown "data-am-dropdown"}
   [:button {:class                   "am-btn am-btn-default am-btn-xs am-dropdown-toggle"
             :data-am-dropdown-toggle "data-am-dropdown-toggle"}
    [:span {:class "am-icon-cog"} "&nbsp;"]
    [:span {:class "am-icon-caret-down"}]]
   [:ul {:class "am-dropdown-content"}
    (for [op (into [operation] operations)
          :let [attr (merge {:href "javascript:void()"} (dissoc op :content))]]
      [:li [:a attr (:content op)]])]])

(defn make-table-row
  "创建hiccup格式表格组件中的一行。
  fields: 放在单元格里面的内容序列。
  last-field: 可选，如果存在，放在最后一列，如果不存在，结果只有fields列。"
  [fields & [last-field]]
  [:tr
   (for [f fields]
     [:td f])
   (if last-field
     [:td last-field])])

(defn make-table
  "创建hiccup格式表格组件。"
  [head-fields row & rows]
  [:table {:class "am-table am-table-bordered am-table-radius am-table-striped am-table-hover"}
   [:thead
    [:tr
     (for [f head-fields]
       [:th f])]]
   [:tbody row rows]])
