(ns yunjia.util.template.classic
  "hiccup风格的html模板库，经典风格。"
  (:require [hiccup.core :refer :all]
            [hiccup.page :refer :all]
            [medley.core :refer [map-vals]]))

(defn make-head
  "创建hiccup格式head组件。
  option-map：选项map，接受如下的key：
    :title 可选。
  styles：样式文件列表。"
  [& [option-map styles]]
  [:head
   [:title (or (:title (map-vals str option-map))
               "")]
   [:meta {:charset "utf-8"}]
   [:meta {:http-equiv "X-UA-Compatible" :content "IE=edge"}]
   [:meta {:name "keywords" :content "index"}]
   [:meta {:name "viewport" :content "width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no"}]
   [:meta {:name "renderer" :content "webkit"}]
   [:meta {:http-equiv "Cache-Control" :content "no-siteapp"}]
   (apply include-css
          "common/assets/css/amazeui.css"
          "common/assets/css/public.css"
          "common/assets/sweetalert/sweetalert.css"
          styles)])

(defn make-footer
  "创建hiccup格式footer组件。
  styles：javascript文件列表。"
  [& scripts]
  [:div
   (apply include-js
          "common/assets/js/jquery.min.js"
          "common/assets/js/amazeui.min.js"
          "common/assets/js/jquery.json.min.js"
          "common/assets/sweetalert/sweetalert.min.js"
          "common/assets/js/public.js"
          scripts)])

(defn- make-menu-1
  "创建一个两级菜单项"
  [root-id index name sub-nodes]
  (let [target-id (format "do-not-say-%d" index)
        data-am-collapse (format "{parent: '#%s', target: '#%s'}" root-id target-id)]
    [:div {:class "am-panel am-panel-default"}
     [:div {:class "am-panel-hd"}
      [:h4 {:class "am-panel-title" :data-am-collapse data-am-collapse}
       [:span {:class "am-icon-subway am-icon-sm"}]
       name]]
     [:div {:id target-id :class "am-panel-collapse am-collapse am-in"}
      [:div {:class "am-panel-bd"}
       [:ul {:class "am-list am-list-static am-list-border"}
        (for [node sub-nodes
              :let [onclick (format "openMenu('%s');" (:path node))
                    sub-name (:name node)]]
          [:li {:onclick onclick} [:span {:class "am-icon-home am-icon-sm"}] sub-name])]]]]))

(defn make-2-level-menu
  "创建hiccup格式两级菜单。"
  [tree name-key children-key & [id]]
  (let [root-id (or id "menu-root-default-id")
        attributes {:class "am-panel-group"
                    :id root-id}
        tree-indexed (map-indexed vector tree)]
    [:div attributes
     (for [[i node] tree-indexed]
       (make-menu-1 root-id i (name-key node) (children-key node)))]))

(defn- fill-id-name
  "填充属性中的id和name。"
  [attributes]
  (let [v (or (:id attributes)
              (:name attributes))]
    (if v
      (merge attributes {:id v :name v})
      attributes)))

(defn make-input
  "创建hiccup格式input组件。
  option-map: 选项map，接受如下的key:
    参考html的元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        random-id (str "input-random-" (rand-int 10000))
        default {:id          random-id
                 :name        random-id
                 :type        "text"
                 :class       "am-form-field"
                 :value       ""
                 :placeholder ""}
        attributes (merge default options)]
    [:input attributes]))

(defn make-textarea
  "创建hiccup格式textarea组件。
  option-map: 选项map，接受如下的key:
    参考html的元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        random-id (str "textarea-random-" (rand-int 10000))
        default {:id    random-id
                 :name  random-id
                 :rows  "6"
                 :cols  "20"
                 :value ""}
        attributes (merge default options)]
    [:textarea (dissoc attributes :value) (:value attributes)]))

(defn make-datepicker
  "创建hiccup格式datepicker组件。
  option-map: 选项map，接受如下的key:
    :format 默认yyyy-mm-dd。
    :setValue 默认2015-01-12。
    其他html元素属性。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))
        o-format (or (:format options)
                     "yyyy-mm-dd")
        o-setValue (or (:setValue options)
                       "2015-01-12")
        data-am-datepicker (format "{format: '%s', setValue: '%s'}" o-format o-setValue)
        random-id (str "datepicker-random-" (rand-int 10000))
        default {:id   random-id
                 :name random-id}
        attributes (-> default
                       (merge options)
                       (dissoc :format :setValue))]
    [:input (merge attributes
                   {:type               "text"
                    :data-am-datepicker data-am-datepicker
                    :readonly           "readonly"})]))

(defn make-input-box
  "创建hiccup格式input组件，存放在div中以调整格式。
  input: 可以是make-input/make-textarea/make-datepicker的输出。
  txt: 标签文本。
  txt-width: 文本宽度，默认为3。"
  [input & [txt txt-width input-width]]
  [:div {:class "am-g am-margin-top"}
   [:div {:class (str "am-u-sm-" (or txt-width "3") " am-text-right")} (or txt "")]
   [:div {:class (str "am-u-sm-" (or input-width "9") " am-u-end col-end")} input]])

(defn make-select
  "创建hiccup格式下拉菜单组件。"
  [& [option-map]]
  (let [options (map-vals str (fill-id-name option-map))]
    )
  (if (:selected map)
    [:option {:value (:value map) :selected "selected"} (:txt map)]
    [:option {:value (:value map)} (:txt map)]
    )
  )

(defn make-form
  "创建hiccup格式form组件。
  option-map: 选项map，接受如下的key:
     :form-type 可选。am-container: 带有padding样式，container: 全屏样式。默认为container。
     其他html元素属性。
  elements: 表单元素组件列表。"
  [option-map element & elements]
  (let [options (map-vals str (fill-id-name option-map))
        form-type (or (:form-type options)
                      "container")
        random-id (str "form-random-" (rand-int 10000))
        default {:id     random-id
                 :name   random-id
                 :class  "am-form"
                 :method "POST"}
        attributes (-> default
                       (merge options)
                       (dissoc :form-type))]
    [:div {:class form-type}
     (into [:form attributes element] elements)]))
