(ns ^{:doc "Database layer for atom feeds" :author "Thomas Engelschmidt"}
  feeds.db
  (:require [clojure.java.jdbc :as sql]
            [clojure.tools.logging :as logging]))

(defmacro log-time
  "Evaluates expr and logs the time it took. Returns the value of
 expr."
  [expr]
  `(let [start# (. System (nanoTime))
         ret# ~expr]
      (logging/debug (str "Elapsed time: " (/ (double (- (. System (nanoTime)) start#)) 1000000.0) " msecs. s-form: " '~expr ))
     ret#))


(defn- clob-to-string [clob]
      "Turn a Derby 10.6.1.0 EmbedClob into a String"
      (when (not (nil? clob))
         (with-open [rdr (java.io.BufferedReader. (.getCharacterStream clob))]
            (apply str (line-seq rdr)))))
 
(defn find-atom-entry [feed id db] 
  (sql/with-connection db
    (sql/transaction                        
     (sql/with-query-results rs ["select id, feed, created_at from atoms where id = ? and feed = ? " id feed]  
        (first rs)))))

(defmacro transform [entry func]
  (if (nil? func) entry `(~func ~entry)))  

(def sql-feed-newest
  "select * from (
   select id, feed,  atom, created_at, seqno from atoms where feed = ?  order by seqno desc )
   where rownum <= ?")

(def sql-feed-next
   "select * from (
      SELECT id, feed, atom, created_at, seqno FROM atoms
      WHERE seqno > ? AND feed = ?
      ORDER BY seqno ASC)
    where rownum <= ?
    ORDER BY seqno DESC")

(def sql-feed-prev
   "select * from (
      SELECT id, feed, atom, created_at, seqno FROM atoms
      WHERE seqno <= ? AND feed = ?
      ORDER BY seqno DESC)
    where rownum <= ?")

(def sql-feed-prev-seq ; seqno < ? - is from the next feed
   "select max (seqno) as seqno from (
      SELECT seqno FROM atoms
      WHERE seqno < ? AND feed = ?
      ORDER BY seqno DESC)
    where rownum <= ?")

(def sql-next-amount
   "select count(*) as amount FROM (
      SELECT feed, seqno FROM atoms
      WHERE seqno > ? AND feed = ?
      ORDER BY seqno DESC)
    where rownum <= ?")

(def sql-prev-amount
   "select count(*) as amount FROM (
      SELECT feed, seqno FROM atoms
      WHERE seqno < ? AND feed = ?
      ORDER BY seqno DESC)
    where rownum <= ?")

(defn load-atom-clj [value]
 (let [str-value  (clob-to-string value)]
  (load-string (str "'" str-value ))))

(defn- transform-map [res merge-into-entry]
  (if (fn? merge-into-entry) 
      (doall (map  (fn [x] (transform (load-atom-clj (:atom x))  merge-into-entry)) res))
      (doall (map  #(load-atom-clj (:atom %)) res))))
                               
(defn find-atom-feed-newest [feed amount merge-into-entry db]
  (logging/debug (str "find-atom-feed-newest args:" feed " " amount " " merge-into-entry " " db))
  (sql/with-connection db
    (let [res (log-time (sql/with-query-results rs [sql-feed-newest feed amount ] (vec rs)))
          via-seqno (:seqno (last res))
          prev-seqno (if (and (not (empty? res)) (>= (count res) amount))
                         (:seqno (log-time (sql/with-query-results rs [sql-feed-prev-seq (int via-seqno) feed amount] (first rs)))))
          trans-res (transform-map res merge-into-entry)]
      [prev-seqno via-seqno trans-res])))

(defn find-atom-feed-with-offset [feed seqno amount merge-into-entry db next?]
  (logging/debug (str "find-atom-feed-with-offset args:" feed " " seqno " " amount " " merge-into-entry " " db " " next?))
  (sql/with-connection db
    (let [res (log-time(sql/with-query-results rs [(if next? sql-feed-next sql-feed-prev) seqno feed amount] (vec rs)))
          next-seqno (if (< 0 (log-time (sql/with-query-results rs [sql-next-amount (:seqno (first res)) feed amount] (:amount (first (vec rs))))))
                           (:seqno (first res)))
          prev-seqno (if (< 0 (log-time (sql/with-query-results rs [sql-prev-amount (:seqno (last res)) feed amount] (:amount (first (vec rs))))))
                           (:seqno (last res)))
          trans-res (transform-map res merge-into-entry)]
      [prev-seqno next-seqno trans-res])))
  
(defn- find-uuid [data] 
   (first (:content 
             (first (filter
                       (fn [x] (= :id (:tag x))) 
                       (:content data))))))

(defn- insert-atom [id feed db col data]
  (let [entry (find-atom-entry feed id db)
        exists (not (empty?  entry))]
    (if exists :conflict 
      (if (< 0 (count (sql/with-connection db
                                  (sql/insert-values :atoms col data)))) :added :not-added))))



(defn insert-atom-entry
  "Insert data into the table"
  ([feed  atom_ db]
   (let [id (find-uuid atom_)]
     (insert-atom id feed db [:id :feed :atom] [id feed (str atom_)]))) 
  ([feed atom_ date db]
   (let [id (find-uuid atom_)]
     (insert-atom id feed db [:id :feed :atom :created_at] [id feed (str atom_) date ])))) 

(defn archive-count [feed db] 
    (log-time (sql/with-connection db
                (sql/with-query-results rs 
                   ["select count(*) as count from atoms where feed =  ?" feed] 
                      (:count  (first (vec rs)))))))
