(ns feeds.atom-reader
  (:use clojure.data.zip.xml)
  (:require [clojure.data.json :as json]
            [clojure.xml :as xml]
            [clojure.zip :as zip]
            [http.async.client :as http-client])
  (:import [java.io ByteArrayInputStream]))

(defn- parse-event [entry]
  (let [category (first (xml-> entry :category (attr :term)))
        content-type (first (xml-> entry :content (attr :type)))
        event (apply merge (map #(assoc {} % (first (xml-> entry % text))) (map :tag (zip/children entry))))]
    (assoc event :category category :content-type content-type)))

(defn- fetch-chunk [url]
  (with-open [client (http-client/create-client)]
    (let [response (http-client/GET client url)
          code (:code (http-client/status response))]
      (http-client/await response)
      (let [response-data (http-client/string response)]
        (when (not (= 200 code))
          (throw (Exception. (str "Error calling feed url: " (http-client/string response)))))
        response-data))))

(defn- get-events [url]
  (let [chunk (fetch-chunk url)
        xml-stream (xml/parse (ByteArrayInputStream. (.getBytes (.trim chunk))))
        zipped (zip/xml-zip xml-stream)
        events (map #(parse-event %) (xml-> zipped :entry))
        next-url (first (xml-> zipped :link [(attr= :ref "next-archive")] (attr :href)))
        prev-url (first (xml-> zipped :link [(attr= :ref "prev-archive")] (attr :href)))]
    [events prev-url next-url]))

(defn- get-head
  "Walks down the feed until the id is found and returns [events next-url] where events is a seq of events starting from the event with id."
  [url id]
  (loop [chunk-url url]
    (if chunk-url
      (let [[events prev-url next-url] (get-events chunk-url)]
        (if (some #(if (= (:id %) id) %) events)
          [(take-while #(not= (:id %) id) events) prev-url next-url]
          (recur prev-url))))))

(defn lazy-atom
  "Returns a lazy sequence starting from the id if provided (otherwise just starting from the url) and ending at the newest event.
   Given that the sequence can be very long and the items fairly large, important to not hold on to the head of the sequence - otherwise
   it cannot be garbage collected along the way."
  [url & [id]]
  (lazy-seq
    (when url
      (let [[events _ next-url] (if id (get-head url id) (get-events url))]
        (concat (reverse events) (lazy-atom next-url))))))

(defn test-lazy []
  (doseq [e (lazy-atom "http://darton.tdk.dk:41001/kunde-v1/sso/notifications/" "c29ea2c1-408c-4d20-b086-544bf59ac180")] (println (:id e) (keys e))))