(ns prismo.core
  (:import [com.couchbase.client.java
            CouchbaseCluster
            CouchbaseAsyncCluster
            Bucket
            AsyncBucket]
           [com.couchbase.client.java.env
            DefaultCouchbaseEnvironment]
           [com.couchbase.client.java.error
            DocumentDoesNotExistException
            DocumentAlreadyExistsException]
           [com.couchbase.client.java.document
            Document BinaryDocument RawJsonDocument StringDocument
            JsonArrayDocument JsonBooleanDocument JsonDocument
            JsonDoubleDocument JsonLongDocument JsonStringDocument
            LegacyDocument SerializableDocument]
           [com.couchbase.client.deps.io.netty.buffer Unpooled UnpooledHeapByteBuf ByteBuf]
           [com.couchbase.client.deps.io.netty.util ResourceLeakDetector ResourceLeakDetector$Level]
           [java.util.concurrent TimeUnit])
  (:require [prismo.rx :as prx]
            [rx.lang.clojure.blocking :as rxb]
            [cheshire.core :as cheshire])
  (:refer-clojure :exclude [empty? get remove]))

(def environment (atom nil))

(defn set-environment [env]
  (when-not (nil? @environment)
    (throw (RuntimeException. "Only one Couchbase environment should exist! Expect stability problems.")))
  (reset! environment env))

(defn get-environment []
  (locking environment
    (when (nil? @environment)
      (reset! environment (-> (DefaultCouchbaseEnvironment/builder)
                            (.bufferPoolingEnabled false)
                            (.build)))))
  @environment)

(defn create-cluster
  "Represents a Couchbase Server Cluster. A Cluster is able to open
  many Buckets while sharing the underlying resources very
  efficiently. Always create only one instance of a CouchbaseCluster
  and share it across threads (same with buckets). The SDK is
  thread-safe, so no additional synchronization is needed when
  interacting with the SDK.

  If different clusters need to be accessed, you MUST reuse the
  ClusterEnvironment. Prismo maintains a singleton
  ClusterEnvironment (see the docstring of `get-environment`), but if
  you create additional environments by other means, you can expect
  timeouts and other stability problems.

  Arguments:
  * `hosts` - a vector of host name or ip address strings."
  [hosts & {:keys [async?]}]
  (if async?
    (CouchbaseAsyncCluster/create (get-environment) hosts)
    (CouchbaseCluster/create (get-environment) hosts)))

;; Cluster stuff
(def disconnect (memfn disconnect))

(defn open-bucket
  ([cluster]
   (.openBucket cluster))
  ([cluster name]
   (.openBucket cluster name))
  ([cluster name password]
   (.openBucket cluster name password))
  ([cluster name password timeout-ms]
   (.openBucket cluster name password timeout-ms TimeUnit/MILLISECONDS)))

(defmacro with-couchbase-cluster [[var & [hosts :as args]] & body]
  `(let [~var (create-cluster ~@args)]
     (try
       ~@body
       (finally
         (disconnect ~var)))))

;; Bucket stuff
(def async (memfn async))
(def close (memfn close))

(defmacro with-open-bucket
  [[var & [& args]] & body]
  `(let [~var (open-bucket ~@args)]
     (try
       ~@body
       (finally
         (close ~var)))))

;;;
;;; Documents
;;;

(defprotocol document
  (content [document])
  (empty? [document])
  (cas [document])
  (expiry [document])
  (id [document]))

(extend-type nil
  document
  (content [d] nil)
  (empty? [d] true)
  (cas [d] nil)
  (expiry [d] nil))

;; JSON documents. We use RawJsonDocument because the Cheshire JSON
;; transcoder is fast and stable and because it saves us from having
;; to mess with the baroque JsonDocument class hierarchy.
;;
;; JSON documents have a special place in Couchbase because they are
;; the only kind of documents that can be used for views and indexes.

(defn- generate-json [obj]
  (cheshire/generate-string obj))

(defn- parse-json [string]
  (cheshire/parse-string string))

(defn make-raw-json-document
  ([id] (when id
          (RawJsonDocument/create id)))
  ([id value & {:keys [cas expiry] :or {cas 0 expiry 0}}]
   (RawJsonDocument/create id expiry (generate-json value) cas)))

(extend-type RawJsonDocument
  document
  (content [d]
    (parse-json (.content d)))
  (id [d]
    (.id d))
  (empty? [d]
    (nil? (.content d)))
  (cas [d]
    (.cas d))
  (expiry [d]
    (.expiry d)))

;; Binary documents

(defn leak-detector-level! [level]
  (ResourceLeakDetector/setLevel
   (case level
     :disabled ResourceLeakDetector$Level/DISABLED
     :simple ResourceLeakDetector$Level/SIMPLE
     :advanced ResourceLeakDetector$Level/ADVANCED
     :paranoid ResourceLeakDetector$Level/PARANOID)))

(def ^:const byte-array-class (class (byte-array 0)))

(defn- byte-array? [a] (instance? byte-array-class a))

(defn- wrapped-buffer [^bytes byte-array]
  (Unpooled/wrappedBuffer byte-array))

(defn- copied-buffer [^bytes byte-array]
  (Unpooled/copiedBuffer byte-array))

(defn ^BinaryDocument make-binary-document
  ([id] (when id
          (BinaryDocument/create id)))
  ([id value & {:keys [cas expiry] :or {cas 0 expiry 0}}]
   (cond
     (byte-array? value) (BinaryDocument/create id expiry (copied-buffer value) cas)
     (instance? UnpooledHeapByteBuf value) (BinaryDocument/create id expiry value cas)
     :else (throw (RuntimeException. "value must be a byte array or ByteBuf")))))

(defn- ^bytes byte-buf-to-byte-array [^ByteBuf bb]
  (if (.hasArray bb)
    (.array bb)
    (let [n (.readableBytes bb)
          ba (byte-array n)]
      (.readBytes bb ba 0 n)
      ba)))

(extend-type BinaryDocument
  document
  (content [d]
    (when-let [^ByteBuf content (.content d)]
      (try
        (byte-buf-to-byte-array content)
        (finally
          (.release content)))))
  (id [d]
    (.id d))
  (empty? [d]
    (nil? (.content d)))
  (cas [d]
    (.cas d))
  (expiry [d]
    (.expiry d)))

;; String documents

(extend-type BinaryDocument
  document
  (content [d]
    (when-let [content (.content d)]
      (byte-buf-to-byte-array content)))
  (id [d]
    (.id d))
  (empty? [d]
    (nil? (.content d)))
  (cas [d]
    (.cas d))
  (expiry [d]
    (.expiry d)))


;;;
;;; More about documents
;;;

(extend-type Document
  document
  (content [d]
    (.content d))
  (id [d]
    (.id d))
  (empty? [d]
    (nil? (.content d)))
  (cas [d]
    (.cas d))
  (expiry [d]
    (.expiry d)))

(defn make-string-document
  ([id] (when id
          (StringDocument/create id)))
  ([id value & {:keys [cas expiry] :or {cas 0 expiry 0}}]
   {:pre [(string? value)]}
   (StringDocument/create id expiry value cas)))

(defn make-document [id value & {:keys [cas expiry type]
                                 :or {cas 0 expiry 0 type nil}}]
  "Make a document, usually a BinaryDocument or RawJsonDocument.

  When the `:type` keyword paramter is nil (the default), this
  function will return either a BinaryDocument (if the value is a byte
  array) or a RawJsonDocument (if the value is anything else that can
  be serialized into JSON by Cheshire.

  It is also possible to set the `:type` keyword parameter to a
  keyword or class in order to make documents of some other types:

   * `:string` or `java.lang.String`: make a StringDocument.

  It would be possible to extend this to other document types, such as
  LegacyDocument or SerializableDocument."
  (let [f (cond
            (#{:string java.lang.String} type) make-string-document
            (#{:binary} type) make-binary-document
            (byte-array? value) make-binary-document
            (#{:raw-json} type) make-raw-json-document
            true make-raw-json-document)]
    (f id value :cas cas :expiry expiry)))

;;
;; Synchronous operations
;;

(defn insert
  "Insert a document.

  In most cases, you will want to use `make-document` to create the
  document.

  The returned Document contains original properties, but has the
  refreshed CAS value set.

  This operation will return successfully if the Document has been
  acknowledged in the managed cache layer on the master server node.

  The `:if-exists` keyword parameter determines what happens if a
  document with the same ID already exists: :error means that a
  DocumentAlreadyExistsException should be thrown;
  :supersede means that the existing document should be replaced; and
  any other value means that the document should not be replaced, an
  exception should not be thrown, but the value of the parameter
  should be returned."
  [bucket document & {:keys [if-exists]
                      :or {if-exists :error}}]
  {:pre [(instance? Document document)
         (not (empty? document))
         (#{:supersede :error} if-exists)]}
  (if (= if-exists :supersede)
    (.upsert bucket document)
    (.insert bucket document)))

(def document-types {:raw-json RawJsonDocument
                     :binary BinaryDocument
                     :string StringDocument
                     ;; Baroque JSON documents from the Java API
                     :json-array JsonArrayDocument
                     :json-boolean JsonBooleanDocument
                     :json JsonDocument
                     :json-double JsonDoubleDocument
                     :json-long JsonLongDocument
                     :json-string JsonStringDocument
                     ;; Compatible with the Couchbase Java API v. 1
                     :legacy LegacyDocument
                     :serializable SerializableDocument})

(defn get
  "Retrieve a document with the specified ID and document type.

  There are two ways to specify the ID and document type:

  1. A (usually empty) document with the same ID and type as the
  document you wish to retrieve.

  2. The ID (in the form of a string) and a document type (either a
  class that implements the Document interface or one of the keys in
  `document-types`) if the document you wish to retrieve.

  Asking for the wrong document type results in a
  com.couchbase.client.java.error.TranscodingException.

  If there is no document with the specified ID, then `nil` is
  returned."
  ([bucket document]
   {:pre [(instance? Document document)]}
   (.get bucket document))
  ([bucket id document-type]
   {:pre [(string? id)
          (or (document-types document-type)
              (.isAssignableFrom Document document-type))]}
   (.get bucket id (if (document-types document-type)
                     (document-types document-type)
                     document-type))))

(defn counter
  "Increment or decrement a counter identified by `id` with the given
  value, `delta`.
  This function may also be called with `initial` and `expiry` arguments
  to specify and initial value for the counter and a TTL.
  "
  ([bucket id delta]
   {:pre [(string? id)]}
   (.counter bucket id delta))
  ([bucket id delta initial]
   {:pre [(string? id)]}
   (.counter bucket id delta initial))
  ([bucket id delta initial expiry]
   {:pre [(string? id)]}
   (.counter bucket id delta initial expiry)))


(defn remove
  "Delete a document by its unique ID.

  The `id-or-document` argument can be either an ID (in the form of a
  string) or a document. If it is a document, then the id is
  extracted from the document.

  If a document with the specified ID does not exist, a
  com.couchbase.client.java.error.DocumentDoesNotExistException will
  be thrown if the keyword argument `:if-does-not-exist` is
  `:error` (the default). If `:if-does-not-exist` is `:ignore`, then
  nil will be returned.

  If the document is successfully removed, a document of unspecified
  type is returned. The Document returned just has the document ID and
  its CAS value set, since the value and all other associated
  properties have been removed from the server.

  "
  [bucket id-or-document & {:keys [if-does-not-exist]
                            :or {if-does-not-exist :error}}]
  {:pre [(or (string? id-or-document)
             (instance? Document id-or-document))
         (#{:error :ignore} if-does-not-exist)]}
  (try
    (.remove bucket id-or-document)
    (catch DocumentDoesNotExistException ex
      (when (= :error if-does-not-exist)
        (throw ex)))))

;;
;; Functions that provide a synchronous interface but use the async
;; library to perform multiple operations in parallel.
;;

(defn- instance-of-any? [classes object]
  ((apply some-fn (map #(partial instance? %1) classes))
   object))

(defn- remove-exceptions [ignorable-classes seq]
  "From a sequence of [a b] pairs, remove any pair for which b is an
  instance of one of the given classes."
  (if (clojure.core/empty? ignorable-classes)
    seq
    (clojure.core/remove (comp (partial instance-of-any?
                                        ignorable-classes)
                               second)
                         seq)))

(defn- group-by-exception [pairs]
  "Turn a sequence of [document-or-id value] pairs into a vector of
  two maps: (1) a map from documents or IDs to results for the
  operations that succeeded and a map from documents or IDs to
  exceptions for the operations that failed."
  (map (partial into {})
       ((juxt clojure.core/remove filter)
        #(instance? Exception (second %))
        pairs)))

(defn- exception? [x]
  (instance? Exception x))

(defn- reorder [keys pairs]
  (let [m (into {} pairs)]
    (map m keys)))

(defn- multi [operator bucket coll
              & {:keys [on-error ignore-exceptions timeout replica]
                 :or {on-error :throw
                      ignore-exceptions []
                      replica nil}}]
  {:pre [(#{:throw :collect} on-error)]}
  (let [ordered-results (->> (prx/multi operator bucket
                                        (clojure.core/remove nil? coll)
                                        :timeout timeout
                                        :replica replica)
                             (rxb/o->seq)
                             (remove-exceptions ignore-exceptions)
                             (reorder coll))]
    (when (and (= on-error :throw)
               (some exception? ordered-results))
      (throw (ex-info (str "Some " operator " operations failed.")
                      {:results ordered-results})))
    ordered-results))

(defn multi-insert [bucket documents & {:keys [on-error if-exists timeout]
                                        :or {on-error :throw
                                             if-exists :error}}]
  {:pre [(#{:throw :collect} on-error)
         (#{:supersede :error} if-exists)]}
  (let [operator (case if-exists
                   :supersede :upsert
                   :error :insert)]
    (multi operator bucket documents
           :on-error on-error :timeout timeout)))

(defn multi-remove [bucket ids-or-documents
                    & {:keys [on-error if-does-not-exist timeout]
                       :or {on-error :throw
                            if-does-not-exist :error}}]
  {:pre [(#{:throw :collect} on-error)
         (#{:error :ignore} if-does-not-exist)]}
  (let [ignorable (case if-does-not-exist
                    :error #{}
                    :ignore #{DocumentDoesNotExistException})]
    (multi :remove bucket ids-or-documents :on-error on-error
           :ignore-exceptions ignorable :timeout timeout)))

(defn multi-get [bucket ids-or-documents & {:keys [on-error timeout replica]
                                            :or {on-error :throw
                                                 replica nil}}]
  {:pre [(#{:throw :collect} on-error)]}
  (multi :get bucket ids-or-documents :on-error on-error :timeout timeout
         :replica replica))

(defn multi-counter [bucket ids & {:keys [on-error timeout]
                                   :or {on-error :throw}}]
  {:pre [(#{:throw :collect} on-error)]}
  (multi :counter bucket ids :on-error on-error :timeout timeout))
