(ns prismo.rx
  (:require [rx.lang.clojure.interop :as rx]
            [clojure.tools.logging :as log])
  (:import [rx Observable]
           [java.util.concurrent TimeUnit]
           [com.couchbase.client.java ReplicaMode AsyncBucket ReplicateTo Bucket]
           [com.couchbase.client.java.document Document]))

(defn- replicate-to [replica]
  (case replica
    :none ReplicateTo/NONE
    :one ReplicateTo/ONE
    :two ReplicateTo/TWO
    :three ReplicateTo/THREE))

(defn- replica-mode [replica]
  (case replica
    :all ReplicaMode/ALL
    :first ReplicaMode/FIRST
    :second ReplicaMode/SECOND
    :third ReplicaMode/THIRD))

(defn- get-multi-operator [operator replica]
  ;; Java methods are not first-class functions, so we make our own.
  (if (nil? replica)
    (case operator
      :insert #(.insert ^AsyncBucket %1 ^Document %2)
      :upsert #(.upsert ^AsyncBucket %1 ^Document %2)
      :remove #(.remove ^AsyncBucket %1 ^Document %2)
      :get #(.get ^AsyncBucket %1 ^Document %2)
      :counter #(.counter ^AsyncBucket %1 ^Document %2 1 1))
    (case operator
      :insert #(.insert ^AsyncBucket %1 ^Document %2 ^ReplicateTo (replicate-to replica))
      :upsert #(.upsert ^AsyncBucket %1 ^Document %2 ^ReplicateTo (replicate-to replica))
      :remove #(.remove ^AsyncBucket %1 ^Document %2 ^ReplicateTo (replicate-to replica))
      :get #(.getFromReplica ^AsyncBucket %1 ^Document %2 ^ReplicaMode (replica-mode replica)))))

(defn ^Observable multi-operation [bucket operator document timeout replica]
  (let [o ((get-multi-operator operator replica) bucket document)]
    (if timeout
      (.timeout ^Observable o ^long timeout TimeUnit/MILLISECONDS)
      o)))

(defn multi
  "Apply an operation to documents or keys (depending on the
  operation) in parallel. Emits vectors, each with two elements: (1) a
  document or key and (2) the result of the operation or an
  exception."
  [operator bucket coll & {:keys [timeout replica]}]
  (-> (Observable/from ^Iterable coll)
      (.flatMap (rx/fn [d]
                  (-> bucket 
                      (#(.async ^Bucket %))
                      (multi-operation operator d timeout replica)
                      (.onErrorReturn (rx/fn [e] e))
                      (.map (rx/fn [v] [d v])))))
      ))
