(ns yetibot.core.adapters.slack
  (:require
    [clojure.core.async :as async]
    [clojure.pprint :refer [pprint]]
    [clojure.core.memoize :as memo]
    [clojure.spec.alpha :as s]
    [yetibot.core.adapters.adapter :as a]
    [robert.bruce :refer [try-try-again] :as rb]
    [clojure.string :as string]
    [yetibot.core.models.users :as users]
    [yetibot.core.util.http :refer [html-decode]]
    [clj-slack
     [users :as slack-users]
     [chat :as slack-chat]
     [channels :as channels]
     [groups :as groups]
     [reactions :as reactions]
     [conversations :as conversations]]
    [slack-rtm.core :as slack]
    [taoensso.timbre :as timbre :refer [color-str debug info error]]
    [yetibot.core.handler :refer [handle-raw]]
    [yetibot.core.chat :refer [base-chat-source chat-source
                               *thread-ts* *target* *adapter*]]
    [yetibot.core.util :as util :refer [image?]]))

(s/def ::type string?)

(s/def ::token string?)

(s/def ::config (s/keys :req-un [::type ::token]))

(def channel-cache-ttl 60000)

(def slack-ping-pong-interval-ms
  "How often to send Slack a ping event to ensure the connection is active"
  10000)

(def slack-ping-pong-timeout-ms
  "How long to wait for a `pong` after sending a `ping` before marking the
   connection as inactive and attempting to restart it"
  5000)

(defn slack-config
  "Transforms yetibot config to expected Slack config"
  [config]
  {:api-url (or (:endpoint config) "https://slack.com/api")
   :token (:token config)})

(defn list-channels [config] (channels/list (slack-config config)))

(def channels-cached
  (memo/ttl
    (comp :channels list-channels)
    :ttl/threshold channel-cache-ttl))

(defn channel-by-id [id config]
  (first (filter #(= id (:id %)) (channels-cached config))))

(defn list-groups [config] (groups/list (slack-config config)))

(defn channels-in
  "all channels that yetibot is a member of"
  [config]
  (filter :is_member (:channels (list-channels config))))

(defn self
  "Slack acount for yetibot from `rtm.start` (represented at (-> @conn :start)).
   You must call `start` in order to define `conn`."
  [conn]
  (-> @conn :start :self))

(defn find-yetibot-user
  [conn cs]
  (let [yetibot-uid (:id (self conn))]
    (users/get-user cs yetibot-uid)))

(defn chan-or-group-name
  "Takes either a channel or a group and returns its name as a String.
   If it's a public channel, # is prepended.
   If it's a group, just return the name."
  [chan-or-grp]
  (str (when (:is_channel chan-or-grp) "#")
       (:name chan-or-grp)))

(defn channels
  "A vector of channels and any private groups by name"
  [config]
  (concat
    (map :name (:groups (list-groups config)))
    (map #(str "#" (:name %)) (channels-in config))))

(defn ->send-msg-options
  "transform message to valid slack/post-message options, including thread
   timestamp when available"
  [msg]
  (let [img? (image? msg)]
    (merge
     {:unfurl_media (str (not (boolean img?))) :as_user "true"}
     (when img?
       {:blocks [{"type" "image"
                  "image_url" msg
                  "alt_text" "Image"}]})
     (when *thread-ts* {:thread_ts *thread-ts*}))))

(defn log-send-msg
  "log the send-msg message and slack's response"
  [msg {:keys [ok] :as response}]
  (let [img? (image? msg)
        resp-str (pr-str response)]
    (debug "send-msg" (color-str :blue {:img? img?
                                        :target *target*
                                        :thread-ts *thread-ts*}))
    (if ok
      (debug "slack response" resp-str)
      (error "error posting to slack" resp-str))))

(defn send-msg
  "defines options based on message and posts a message to slack, with some
    additional logging"
  [config msg]
  (let [cfg (slack-config config)
        opts (->send-msg-options msg)
        resp (slack-chat/post-message cfg
                                      *target*
                                      msg
                                      opts)]
    (log-send-msg msg resp)))

(defn send-paste [config msg]
  (slack-chat/post-message
    (slack-config config) *target* ""
    (merge
      {:unfurl_media "false" :as_user "true"
       :attachments [{:pretext "" :text msg}]}
      (when *thread-ts* {:thread_ts *thread-ts*}))))

;; formatting

(defn unencode-message
  "Slack gets fancy with URL detection, channel names, user mentions, as
   described in https://api.slack.com/docs/formatting. This can break support
   for things where YB is expecting a URL (e.g. configuring Jenkins), so strip
   it for now. Replaces <X|Y> with Y.

   Secondly, as of 2017ish first half, Slack started mysteriously encoding
   @here and @channel as <!here> and <!channel>. Wat.

   <!here> becomes @here
   <!channel> becomes @channel
   <!subteam^S03> becomes @S03 (which isn't helpful - we'd have to look up the
                                subteam to get the friendly name)

   Why are you gross, Slack"
  [body]
  (-> body
    (string/replace  #"\<\!(.+)\>" "@$1")
    (string/replace #"\<(.+)\|(\S+)\>" "$2")
    (string/replace #"\<(\S+)>" "$1")
    html-decode))

(comment
  (unencode-message "<!subteam^S04FJDES3H6>")
  )

(defn entity-with-name-by-id
  "Takes a message event and translates a channel ID, group ID, or user id from
   a direct message (e.g. 'C12312', 'G123123', 'D123123') into a [name entity]
   pair. Channels have a # prefix"
  [config event]
  (info "entity-with-name-by-id" (pr-str {:event event}))
  (let [sc (slack-config config)
        chan-id (:channel event)]
    (condp = (first chan-id)
      ;; direct message - lookup the user
      \D (let [e (:user (slack-users/info sc (:user event)))]
           [(:name e) e])
      ;; channel, group or anything else, lookup the converstaion
      (let [e (:channel (conversations/info sc chan-id))]
           [(str "#" (:name e)) e]))))

(defn fetch-users [config]
  (let [sc (slack-config config)]
    (->> (slack-users/list sc)
         :members
         (filter #(not (:deleted %))))))

;; events

(defn on-channel-join
  "reaction to YB user 'channel join' events, including binding the user's
   chat source with the related channel, and sending to the handler"
  [{:keys [channel] :as e} conn config]
  (let [[chan-name _entity] (entity-with-name-by-id config {:channel channel})
        cs (chat-source chan-name)
        user-model (users/get-user cs (:user e))
        yetibot-user (find-yetibot-user conn cs)]
    (binding [*target* channel]
      (timbre/info "channel join" (color-str :blue (with-out-str (pprint cs))))
      (handle-raw cs user-model :enter yetibot-user {}))))

(defn on-channel-leave
  "reaction to YB user 'channel leave' events, including binding the user's
   chat source with the related channel, and sending to the handler"
  [{:keys [channel] :as e} conn config]
  (let [[chan-name _entity] (entity-with-name-by-id config {:channel channel})
        cs (chat-source chan-name)
        user-model (users/get-user cs (:user e))
        yetibot-user (find-yetibot-user conn cs)]
    (binding [*target* channel]
      (timbre/info "channel leave" e)
      (handle-raw cs user-model :leave yetibot-user {}))))

(defn on-message-changed
  "reaction to when a message has changed, but ignoring changes from the YB
   user, and sending to the handler"
  [{:keys [channel] :as event
    {:keys [user text thread_ts]} :message}
   conn config]
  (timbre/info "message changed" \newline (pr-str event))
  ;; ignore message changed events from Yetibot - it's probably just Slack
  ;; unfurling stuff and we need to ignore it or it will result in double
  ;; history
  (let [[chan-name _entity] (entity-with-name-by-id config {:channel channel
                                                            :user user})
        cs (chat-source chan-name)
        yetibot-user (find-yetibot-user conn cs)
        yetibot-uid (:id yetibot-user)
        yetibot? (= yetibot-uid user)
        user-model (assoc (users/get-user cs user)
                          :yetibot? yetibot?)]
    (if yetibot?
      (info "ignoring message changed event from Yetibot user" yetibot-uid)
      (binding [*target* channel
                *thread-ts* thread_ts]
        (handle-raw cs
                    user-model
                    :message
                    yetibot-user
                    {:body (unencode-message text)})))))

(defn on-message
  "reaction to any message, dispatching to the appropriate event handler
   after determing the event sub-type, while taking into consideration
   whether the message is consider a 'bot_message'"
  [{:keys [conn config]}
   {:keys [subtype ts]
    chan-id :channel
    thread-ts :thread_ts :as event}]
  (info "on-message" (color-str :blue (pr-str event)))
  ;; allow bot_message events to be treated as normal messages
  (if (and (not= "bot_message" subtype) subtype)
    ; handle subtypes
    (do
      (info "event subtype" subtype)
      ; handle the subtype
      (condp = subtype
        "channel_join" (on-channel-join event conn config)
        "group_join" (on-channel-join event conn config)
        "channel_leave" (on-channel-leave event conn config)
        "group_leave" (on-channel-leave event conn config)
        "message_changed" (on-message-changed event conn config)
        ; do nothing if we don't understand
        (info "Don't know how to handle message subtype" subtype)))

    ; handle normal messages
    (let [[chan-name entity] (entity-with-name-by-id config event)
          ;; _ (info "channel entity:" (pr-str entity))
          ;; _ (info "event entity:" (color-str :red (pr-str event)))
          ;; TODO we probably need to switch to chan-id when building the
          ;; Slack chat-source since they are moving away from being able to
          ;; use user names as IDs
          cs (assoc (chat-source chan-name)
                    ;; allow command handlers access to the raw event in case
                    ;; they need platform-specific data e.g. in slack's case
                    ;; they encode emoji in the text but the original event has
                    ;; the actual unicode
                    :raw-event event
                    :is-private (:is_private entity))
          _ (info "chat source"
                  (color-str :green (pr-str {:entity entity
                                             :chat-source cs})))
          yetibot-user (find-yetibot-user conn cs)
          yetibot-uid (:id yetibot-user)
          yetibot? (= yetibot-uid (:user event))
          user-model (assoc (users/get-user cs (:user event))
                            :yetibot? yetibot?)
          body (if (string/blank? (:text event))
                 ;; if text is blank attempt to read an attachment fallback
                 (->> event :attachments (map :text) (string/join \newline))
                 ;; else use the much more common `text` property
                 (:text event))]
      (info (color-str :green {:thread-ts thread-ts :ts ts}))

      (binding [*thread-ts* thread-ts
                *target* chan-id]
        (handle-raw cs
                    user-model
                    :message
                    yetibot-user
                    {:body (unencode-message body)
                     ;; allow chatters (like obs) to optionally reply in-thread
                     ;; by propagating the thread-ts
                     :thread-ts (or thread-ts ts)})))))


(defn on-reaction-added
  "reaction related to when a user adds an emoji to an item:
   https://api.slack.com/events/reaction_added"
  [{:keys [conn config]}
   {:keys [reaction item_user item] :as event}]
  (let [[chan-name _entity] (entity-with-name-by-id config item)
        sc (slack-config config)
        cs (assoc (chat-source chan-name)
                  :raw-event event)
        yetibot-user (find-yetibot-user conn cs)
        yetibot-uid (:id yetibot-user)
        yetibot? (= yetibot-uid (:user event))
        user-model (assoc (users/get-user cs (:user event))
                          :yetibot? yetibot?)
        reaction-message-user (assoc (users/get-user cs item_user)
                                     :yetibot? (= yetibot-uid item_user))

        {[parent-message] :messages} (conversations/history
                                      sc (:channel item)
                                      {:latest (:ts item)
                                       :inclusive "true"
                                       :count "1"})

        parent-ts (:ts parent-message)

        ;; figure out if the user reacted to the top level parent of the thread
        ;; or a child
        is-parent? (= (:ts parent-message) (:ts item))

        child-ts (:ts item)

        child-message (when-not is-parent?
                        (->> (conversations/replies
                              sc (:channel item)
                              parent-ts
                              {:latest child-ts
                               :inclusive "true"
                               :limit "1"})
                             :messages
                             (filter (fn [{:keys [ts]}] (= ts child-ts)))
                             first))

        message (if is-parent? parent-message child-message)]
    ;; only support reactions on message types
    (when (= "message" (:type item))
      (binding [*target* (:channel item)
                *thread-ts* parent-ts]

        (when (string/includes? reaction "delete")
          (info "delete this message" event)
          (slack-chat/delete
           sc (:ts message) (:channel item)))

        (handle-raw cs user-model :react yetibot-user
                    {:reaction (string/replace reaction "_" " ")
                     ;; body of the message reacted to
                     :body (:text message)
                     ;; user of the message that was reacted to
                     :message-user reaction-message-user})))))

(defn on-hello [event]
  (timbre/debug "Hello, you are connected to Slack" event))

(defn on-pong
  "handler related to when pong events are received"
  [{:keys [connection-last-active-timestamp connection-latency ping-time]}
   event]
  (let [ts @ping-time
        now (System/currentTimeMillis)]
    (timbre/trace "Pong" (pr-str event))
    (reset! connection-last-active-timestamp now)
    (when ts (reset! connection-latency (- now ts)))))

(defn start-pinger!
  "Send a ping event to Slack to ensure the connection is active"
  [{:keys [conn should-ping? ping-time]}]
  (async/go-loop [n 0]
    (when @should-ping?
      (when-let [c @conn]
        (let [ts (System/currentTimeMillis)
              ping-event {:type :ping
                          :id n
                          :time ts}]
          (timbre/trace "Pinging Slack" (pr-str ping-event))
          (reset! ping-time ts)
          (slack/send-event (:dispatcher c) ping-event)
          (async/<!! (async/timeout slack-ping-pong-interval-ms))
          (recur (inc n)))))))

(defn stop-pinger! [{:keys [should-ping?]}]
  (reset! should-ping? false))

(defn on-connect
  "handler related to when slack is connected, starting the slack pinger
   service"
  [{:keys [connected? should-ping?] :as adapter} _event]
  (reset! should-ping? true)
  (reset! connected? true)
  (start-pinger! adapter))

(declare restart)

;; See https://developer.mozilla.org/en-US/docs/Web/API/CloseEvent for the full
;; list of status codes on a close event
(def status-normal-close 1000)

(defn on-close
  "handler related to when the slack connection has been closed, determing
   if this was intentional, and if not intentional, to retry a connection"
  [{:keys [config connected?] :as adapter}
   {:keys [status-code] :as status}]
  (reset! connected? false)
  (timbre/info "close" (:name config) status)
  (when (not= status-normal-close status-code)
    (try-try-again
      {:decay 1.1 :sleep 5000 :tries 500}
      (fn []
        (timbre/info "attempt no." rb/*try* " to rereconnect to slack")
        (when rb/*error* (timbre/info "previous attempt errored:" rb/*error*))
        (when rb/*last-try* (timbre/warn "this is the last attempt"))
        (restart adapter)))))

(defn on-error [exception]
  (timbre/error "error in slack" exception))

(defn handle-presence-change [e]
  (let [active? (= "active" (:presence e))
        id (:user e)
        source (select-keys (base-chat-source) [:adapter])]
    (when active?
      (debug "User is active:" (pr-str e)))
    (users/update-user source id {:active? active?})))

(defn on-presence-change [e]
  (handle-presence-change e))

(defn on-manual-presence-change [e]
  (timbre/debug "manual presence changed" e)
  (handle-presence-change e))

(defn on-channel-joined
  "Fires when yetibot gets invited and joins a channel or group"
  [e]
  (timbre/debug "channel joined" e)
  (let [c (:channel e)
        cs (chat-source (:id c))
        user-ids (:members c)]
    (timbre/debug "adding chat source" cs "for users" user-ids)
    (run! #(users/add-chat-source-to-user cs %) user-ids)))

(defn on-channel-left
  "Fires when yetibot gets kicked from a channel or group"
  [e]
  (timbre/debug "channel left" e)
  (let [c (:channel e)
        cs (chat-source c)
        users-in-chan (users/get-users cs)]
    (timbre/debug "remove users from" cs (map :id users-in-chan))
    (run! #(users/remove-user cs (:id %)) users-in-chan)))

;; users

(defn filter-chans-or-grps-containing-user [user-id chans-or-grps]
  (filter #((-> % :members set) user-id) chans-or-grps))

(defn reset-users
  "Previously we tried to capture which channels/groups each user was in. We no
  longer have that data from Slack on startup, so just store the users
  themselves without any channels/groups."
  [users]
  (run!
    (fn [user]
      (let [user-id (:id user)
            username (:name user)
            mention-name (str "<@" (:id user) ">")
            user-model (users/create-user
                         username
                         (assoc user :mention-name mention-name))]
        (users/add-user-without-channel
          (:adapter (base-chat-source))
          user-model)))
    users))

;; lifecycle

(defn stop [{:keys [should-ping? conn] :as adapter}]
  (timbre/info "Stop Slack" (pr-str should-ping?))
  (reset! should-ping? false)
  (when @conn
    (timbre/info "Closing Slack" (a/uuid adapter))
    (slack/send-event (:dispatcher @conn) :close))
  (reset! conn nil))

(defn restart
  "conn is a reference to an atom.
  config is a map"
  [{:keys [conn config] :as adapter}]
  (reset! conn (slack/start (slack-config config)
                            :on-connect (partial #'on-connect adapter)
                            :on-error on-error
                            :on-close (partial #'on-close adapter)
                            :presence_change on-presence-change
                            :channel_joined on-channel-joined
                            :group_joined on-channel-joined
                            :channel_left on-channel-left
                            :group_left on-channel-left
                            :manual_presence_change on-manual-presence-change
                            :message (partial #'on-message adapter)
                            :reaction_added (partial #'on-reaction-added adapter)
                            :pong (partial #'on-pong adapter)
                            :hello on-hello))
  (info "Slack (re)connected as Yetibot with id" (:id (self conn)))
  (let [users (fetch-users (slack-config config))]
    (info "Resetting users:" (pr-str (count users)))
    (reset-users users)))

(defn start
  "start the slack connection service"
  [adapter _conn config _connected?]
  (stop adapter)
  (binding [*adapter* adapter]
    (info "adapter" adapter "starting up with config" config)
    (restart adapter)))

(defn history
  "chan-id can be the ID of a:
   - channel
   - group
   - direct message
   Retrieve history from the correct corresponding API."
  [adapter chan-id]
  (let [c (slack-config (:config adapter))]
    (conversations/history c chan-id)))

(defn react
  "reacts to messages in a given channel for a specific adapter that are
   non-YB commands and not from the YB user"
  [adapter emoji channel]
  (info "react" {:adapter adapter :emoji emoji :channel channel})
  (let [c (slack-config (:config adapter))
        conn (:conn adapter)
        yb-id (:id (self conn))
        hist (history adapter channel)
        non-yb-non-cmd (->> (:messages hist)
                            (filter #(not= yb-id (:user %)))
                            (filter #(not (string/starts-with? (:text %) "!"))))
        msg (first non-yb-non-cmd)
        ts (:ts msg)]
    (debug "react with"
           (pr-str
            {:msg msg :emoji emoji :channel channel :timestamp ts}))
    (reactions/add c emoji {:channel channel :timestamp ts})))

;; adapter impl

(defrecord Slack [config conn connected? connection-last-active-timestamp
                  ping-time connection-latency should-ping?]
  a/Adapter

  (a/uuid [_] (:name config))

  (a/platform-name [_] "Slack")

  (a/channels [_] (channels config))

  (a/send-paste [_ msg] (send-paste config msg))

  (a/send-msg [_ msg] (send-msg config msg))

  (a/join [_ _channel]
    (str
      "Slack bots such as myself can't join channels on their own. Use /invite "
      "@yetibot from the channel you'd like me to join instead.✌️"))

  (a/leave [_ _channel]
    (str "Slack bots such as myself can't leave channels on their own. Use /kick "
         "@yetibot from the channel you'd like me to leave instead. 👊"))

  (a/chat-source [_ channel] (chat-source channel))

  (a/stop [adapter] (stop adapter))

  (a/connected? [{:keys [connected?
                         connection-last-active-timestamp]}]
    (when @connection-last-active-timestamp
      (let [now (System/currentTimeMillis)
           time-since-last-active (- now @connection-last-active-timestamp)
           surpassed-timeout? (> time-since-last-active
                                 (+ slack-ping-pong-timeout-ms
                                    slack-ping-pong-interval-ms))]
       (and @connected?
            (not surpassed-timeout?)))))


  (a/connection-last-active-timestamp [_] @connection-last-active-timestamp)

  (a/connection-latency [_] @connection-latency)

  (a/start [adapter] (start adapter conn config connected?)))

(defn make-slack
  [config]
  (map->Slack
    {:config config
     :conn (atom nil)
     :connected? (atom false)
     :connection-latency (atom nil)
     :connection-last-active-timestamp (atom nil)
     :ping-time (atom nil)
     :should-ping? (atom false)}))
