(ns warpcore.db.core
  (:require [warpcore.util.core :as util]
            [datomic.api :as d]
            [taoensso.timbre :as timbre]
            [environ.core :refer [env]]))

(timbre/refer-timbre)

(def ^:dynamic *datomic-uri* (env :datomic-uri))

(defn conn [] (d/connect *datomic-uri*))

(defn db [] (d/db (conn)))

;; route helper
(defn wrap-db
  "A Ring middleware that provides a request-consistent database connection and
  value for the life of a request."
  [handler]
  (fn [request]
    (let [conn (conn)]
      (handler (assoc request
                 :conn conn
                 :db (d/db conn))))))

;; creation
(defn load-dtm [conn filename]
  (info "Loading schema file: " filename)
  ;; parse and submit schema transaction
  @(d/transact conn (read-string (slurp (clojure.java.io/resource filename)))))

(defn create! []
  ;; create database
  (info "Creating database" *datomic-uri*)
  (d/create-database *datomic-uri*)
  (let [conn (conn)] (load-dtm conn "db/schema.dtm") conn))

;; models

(def meta-props [:uuid :created-at :link-id])

(defn sanitize [props allowed-props & [disallowed-props]]
  (apply dissoc (select-keys props allowed-props)
         (concat meta-props disallowed-props)))

(defn delete! [conn uuid]
  @(d/transact conn [[:db.fn/retractEntity [:uuid uuid]]]))

(defn save! [conn entity &
             [{:keys [fill-attrs
                      blacklist-props
                      whitelist-props
                      temp-id] :as opts}]]
  (let [{:keys [created-at? uuid?]} fill-attrs
        tx (merge (if (or whitelist-props blacklist-props)
                    (sanitize entity whitelist-props
                              (concat meta-props blacklist-props))
                    entity)
                  (util/remove-nil-vals {:db/id (or temp-id (d/tempid :db.part/user))
                                         :uuid (if uuid? (d/squuid))
                                         :created-at (if created-at? (java.util.Date.))}))]
    @(d/transact conn [(util/remove-nil-vals tx)])))

(defn save-and-return-full! [conn entity &
                             [{:keys [pull-props] :as opts}]]
  (debug "Creating entity:" entity)
  (let [id (d/tempid :db.part/user)
        {tempids :tempids dba :db-after}
        (save! conn entity (assoc opts :temp-id id))]
    {:db-after dba
     :new-entity
     (d/pull dba (or (set pull-props) "[*]")
             (d/resolve-tempid dba tempids id))}))

(defn save-and-return! [conn entity & [opts]]
  (:new-entity (save-and-return-full! conn entity opts)))
