(ns warpcore.core
  (:require [dbutil.core :as db]
            [org.httpkit.server]
            [compojure
             [core :refer [GET POST PUT DELETE defroutes]]
             [route :refer [resources]]]
            [ring.middleware defaults stacktrace reload]
            [environ.core :refer [env]]
            [clojure.tools.nrepl.server :as nrepl]
            [datomic.api :as d]
            [taoensso.sente :as sente]
            [taoensso.sente.server-adapters.http-kit]
            [taoensso.timbre :as timbre]
            [warpcore.fb :as fb]
            [warpcore.emailqueue :as email]))

(timbre/refer-timbre)

(defn timbre-config! []
  (timbre/set-config! [:shared-appender-config :spit-filename] "log/app.log")
  (timbre/set-config! [:appenders :spit :enabled?] true))

(defmacro sente-setup []
  `(do (defonce ~'sente-init
         (let [{ch-recv# :ch-recv
                send-fn# :send-fn
                ajax-post-fn# :ajax-post-fn
                ajax-get-or-ws-handshake-fn# :ajax-get-or-ws-handshake-fn
                connected-uids# :connected-uids}
               (taoensso.sente/make-channel-socket!
                taoensso.sente.server-adapters.http-kit/http-kit-adapter
                {:user-id-fn #(or (warpcore.fb/user-id %)
                                  (get-in % [:session :temp-id]))})]

           (def ~'ring-ajax-post ajax-post-fn#)
           (def ~'ring-ajax-get-or-ws-handshake ajax-get-or-ws-handshake-fn#)
           (def ~'ch-recv ch-recv#) ; ChannelSocket's receive channel
           (def ~'chsk-send! send-fn#) ; ChannelSocket's send API fn
           (def ~'connected-uids connected-uids#) ; Watchable, read-only atom
           ))

       (defmulti ~'event-msg-handler :id)

       (defmethod ~'event-msg-handler :default ; Fallback
         [{:as ~'ev-msg
           ring-req# :ring-req
           ?data# :?data
           event# :event
           id# :id
           ?reply-fn# :reply-fn
           send-fn# :send-fn}]
         (let [session# (:session ring-req#)
               nickname# (:nickname session#)]
           (info "Unhandled event: %s" event#)
           (when ?reply-fn#
             (?reply-fn# {:umatched-event-as-echoed-from-from-server event#}))))

       (defmethod ~'event-msg-handler :chsk/ws-ping [_#]
         (debug "Ping received."))

       (defmethod ~'event-msg-handler :chsk/uidport-open [_#]
         (info "Port opened"))

       (defmethod ~'event-msg-handler :chsk/uidport-close [_#]
         (info "Port closed"))

       (defonce ~'chsk-router
         (taoensso.sente/start-chsk-router! ~'ch-recv #'~'event-msg-handler))))

(defn start-nrepl-server! []
  (let [port (env :nrepl-port 7888)]
    (info "Starting nrepl server on port" port)
    (defonce nrepl-server
      (nrepl/start-server :port port))))

(defn start-http-server! [handler]
  (let [port (env :port 8080)]
    (info "Starting web server on port" port)
    (defonce http-server
      (org.httpkit.server/run-server handler {:port port :join? false}))))

(defn wrap-defaults [handler]
  (let [ring-defaults-config
        (assoc-in ring.middleware.defaults/site-defaults [:security :anti-forgery]
                  {:read-token (fn [req] (-> req :params :csrf-token))})]
    (-> handler
        (ring.middleware.defaults/wrap-defaults ring-defaults-config)
        db/wrap-db
        ring.middleware.stacktrace/wrap-stacktrace)))

(defn boot! [opts]
  (timbre-config!)
  (when (:nrepl opts)
    (start-nrepl-server!))

  (when (:datomic opts)
    (db/create!)
    (when-not (env :lein-no-dev)
      (db/load-dtm (db/conn) "db/fixtures.dtm")))

  (when (:email opts)
    (email/init!))

  (when (:http opts)
    (start-http-server! (get-in opts [:http :handler]))))
