(ns optimus.class-path
  (:require [clojure.string :as s]
            [clojure.java.io :as io])
  (:import [java.io File]
           [java.net URL]
           [java.util.zip ZipFile ZipEntry]))

;; there are major performance improvements to be gained by not
;; traversing the entirety of the class path when running locally and
;; picking up files from the class path for every request. Since
;; you're not serving files from a .m2 folder in production, this is
;; hopefully a safe bet.

(defn- parse-jar-url
  [^java.net.URL url]
  (let [^String p (.getPath url)
        idx (.lastIndexOf p "!")]
    (vector
      (io/file
        (if (neg? idx)
          p
          (if (.startsWith p "file:")
            (subs p 5 idx)
            (subs p 0 idx))))
      (when-not (neg? idx)
        (subs p (inc idx))))))

(defn- list-jar-contents
  [^URL url]
  (let [[jar-file path] (parse-jar-url url)]
    (when (.isFile jar-file)
      (let [contents (->> (ZipFile. jar-file)
                          (.entries)
                          (enumeration-seq)
                          (mapv #(.getName ^ZipEntry %)))
            p (when path
                (if (.startsWith ^String path "/")
                  (subs path 1)
                  path))]
        (if p
          (filter #(.startsWith ^String % p) contents)
          contents)))))

(defn- list-files
  [^URL url]
  (letfn [(lazy-traverse [^File dir]
            (lazy-seq
              (let [fs (.listFiles dir)]
                (mapcat
                  (fn [^File f]
                    (if (.isDirectory f)
                      (lazy-traverse f)
                      [(.getCanonicalPath f)]))
                  fs))))]
    (let [^File f (io/file url)]
      (if (.isDirectory f)
        (lazy-traverse f)
        [(.getCanonicalPath f)]))))

(defn resource-paths
  [directory-path]
  (when-let [^URL url (io/resource directory-path)]
    (case (.getProtocol url)
      "jar"  (list-jar-contents url)
      "file" (list-files url)
      nil)))
