(ns xafizoff.ac
  (:require [goog.string :as gstr])
  (:import (goog.ui.ac AutoComplete Renderer InputHandler)))

(def ac-map (atom {}))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;; MATCHER ;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defprotocol IACSuggest
  "Protocol for suggestion source"
  (suggest [this token max-matches]))

(extend-type function
  IACSuggest
  (suggest [f token max-matches]
    (f token max-matches)))

(extend-type array
  IACSuggest
  (suggest [a token max-matches]
    (->> a
         seq
         (filter #(gstr/startsWith % token))
         (take max-matches)
         into-array)))

(extend-type string
  IACSuggest
  (suggest [s token max-matches]))

(defrecord Matcher [src]
  Object
  (requestMatchingRows [this token max-matches match-handler]
    (match-handler token (suggest src token max-matches))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;; SELECTION HANDLER ;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- parse-token [value caret]
  (let [delim (set ".,/#!$%^&*;:+?- \n")]
    (->> (subs value 0 caret)
         reverse
         (take-while #(not (contains? delim %)))
         reverse
         (apply str))))

(defn SelectionHandler
  "Selection handler for auto completion.
  Extends goog.ui.ac.InputHandler"
  []
  (.call InputHandler (js* "this") nil nil true nil))
(goog/inherits SelectionHandler InputHandler)
(set! (.. SelectionHandler -prototype -setTokenText)
      (fn [token-text _]
        (let [this (js* "this")
              caret (.getCursorPosition this)
              value (.getValue this)
              elem (.getActiveElement this)
              start (->> (parse-token value caret)
                         count
                         (- caret))
              new-val (str (subs value 0 start) token-text)]
          (.setValue this new-val)
          (swap! ac-map #(assoc % elem (-> % (get elem)
                                           (assoc :prev new-val))))
          (set! (.-rowJustSelected_ this) true)
          )))
(set! (.. SelectionHandler -prototype -parseToken)
      (fn []
        (let [this (js* "this")
              caret (.getCursorPosition this)
              value (.getValue this)]
          (parse-token value caret))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;   AUTO COMPLETE ;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ^:export wrap
  "Autocomplete for inputs using goog.ui.ac.Autocomplete"
  ([input suggest]
   (wrap input nil suggest))
  ([input suggest-area suggest]
   (js/console.log "Initializing autocomplete....")
   (let [ac- (some-> @ac-map (get input) :ac)]
     (when ac-
       (.dispose ac-)
       (swap! ac-map #(dissoc % input)))
     )
   (let [matcher (Matcher. suggest)
         renderer (Renderer. suggest-area)
         handler (SelectionHandler.)
         auto-complete (AutoComplete. matcher renderer handler)]
     (.attachAutoComplete handler auto-complete)
     (.attachInput handler input)
     (swap! ac-map #(assoc % input {:ac auto-complete}))
     auto-complete)))