(ns convex.db

  "Etch is a fast, immutable, embedded database tailored for cells.

   It can be understood as a data store where keys are hashes of the cells they point to.
   Hence, the API is pretty simple. [[write]] takes a cell and returns a hash while [[read]]
   takes a hash and returns a cell (or nil if not found).

   Most of the time, usage is even simpler by using [[root-write]] and [[root-read]] to persist
   the state of a whole application at once (only new values are effectively written).

   Data is retrieved semi-lazily. For instance, in the case of a large vector, only the 
   the top structure of that vector is fetched. Elements are read from disk when actually accessed
   then cached using a clever system of soft references under the hood. This explains why data
   larger than memory can be retrieved and handled since the JVM can garbage-collect those soft
   references ; their value will be read from disk again if required. Nonetheless, users only deal
   with cells and all this process is completely transparent.

   Attention, although this namespace is straightforward, one rule must be followed at all time:
   cells read from an instance can only be written back to that instance. In other words, one
   must never mix cells read from different instances with the intent of writing them anywhere.
   This will result in some of the data not being written. Everything, from cells to Etch, has
   been heavily optimized for Convex peers that only ever handle 1 instance at a time. It is
   fine using several stores in the same process as long as operations never cross-over.

   Convex tooling, whenever an instance is needed, will always look for the instance associated 
   with the current thread (if any). The typical workflow is to call [[current-set]] after [[open]]:

   ```clojure
   (convex.db/current-set (convex.db/open \"my/instance.etch\"))
   (convex.db/read (convex.db/write (convex.cell/* [:a :b 42])))
   (convex.db/close)
   ```

   If no instance is bound to the current thread explicitely, a temporary one is created whenever needed.
   See [[global-set]] for improving the workflow when an instance is needed in more than one thread.
  
   When using a [[convex.cvm/ctx]], its state is initially hold in memory. After opening an Etch
   instance and setting it as thread-local, this state can be retrieved at any point using [[convex.cvm/state]]
   and persisted to disk since it is a cell. This renders that state garbage-collecteable as exposed
   above. Of course, it is important not to close the instance before stopping all operations on that
   context and its state."

  {:author "Adam Helinski"}

  (:import (convex.core.data ACell
                             Hash)
           (convex.core.store Stores)
           (java.io File)
           (etch EtchStore))
  (:refer-clojure :exclude [flush
                            read]))


(set! *warn-on-reflection*
      true)


;;;;;;;;;;


(defn current

  "Returns the thread-local instance (or nil).
   See [[current-set]]."

  ^EtchStore

  []

  (Stores/current))



(defn current-set

  "Binds the given `instance` to the current thread.
   Returns the `instance`.
   See [[current]]."

  ^EtchStore

  [instance]

  (Stores/setCurrent instance)
  instance)



(defn global-set

  "When an instance is used in more than one thread, it is a good idea using this function.
   Convex tooling will then use the given `instance` in all thread automatically unless it is
   overwritten with [[current-set]] on a thread per thread basis."

  ^EtchStore

  [instance]

  (Stores/setGlobalStore instance)
  instance)


;;;;;;;;;; Lifecycle


(defn close

  "Flushes and closes the thread-local instance. Also unbinds it from the current thread.
   
   Note that all instances are also cleanly closed on JVM shutdown but it is
   more predictable doing it manually."

  []

  (.close (current))
  (current-set nil)
  nil)



(defn flush

  "Flushes the thread-local instance, ensuring all changes are persisted to disk."

  ^EtchStore

  []

  (.flush (current))
  nil)



(defn open

  "Opens an instance at the given `path`.
   File is created if needed."

  ^EtchStore

  [^String path]

  (let [^File file (File. path)]
    (-> file
        (.getParentFile)
        (.mkdirs))
    (EtchStore/create file)))



(defn open-tmp

  "Like [[open]] but creates a temporary file.
   A prefix string may be provided for the filename."


  (^EtchStore []

   (EtchStore/createTemp))


  (^EtchStore [prefix]

   (EtchStore/createTemp prefix)))



(defn path

  "Returns the path of thread-local instance."

  []

  (.getFileName (current)))


;;;;;;;;;; R\W


(defn read

  "Reads from the thread-local instance and returns the cell for the given `hash` (or nil
   if not found)."
  
  ^ACell
  
  [^Hash hash]

  (some-> (.refForHash (current)
                       hash)
          (.getValue)))



(defn write

  "Writes the given `cell` to the thread-local instance and returns its hash.

   Very basic cells are not persisted because that would be inefficient and hardly ever happens.
   They are typically embedded in collections. Hence, this function will return nil for:

     - Address
     - Empty collections
     - Primitives (boolean, byte, double, long)
     - Symbolic (keywords and symbols)"

  ^Hash

  [^ACell cell]
   
  (-> (ACell/createPersisted cell)
      (.cachedHash)))


;;;


(defn root-read

  "Returns the cell stored at the root of the thread-local instance.

   The root is a place in the instance that can be read without providing a hash. It is commonly
   used for storing the whole state of an application or at least some sort of index containing
   hashes of other data in the instance. This makes Etch self-sufficient as no hash must be stored
   externally.

   See [[write-root]]."

  ^ACell

  []

  (read (.getRootHash (current))))



(defn root-write

  "Writes the given `cell` to the root of the thread-local instance and returns its hash.
   Behaves like [[write]].

   See [[read-root]]."

  ^Hash

  [^ACell cell]

  (when-some [hash (write cell)]
    (.setRootHash (.getEtch (current))
                  hash)
    hash))
