(ns works-hub.pachyderm.customers
  (:refer-clojure :exclude [list update])
  (:require [works-hub.pachyderm.subscriptions :as subscriptions]
            [works-hub.pachyderm.transforms :as trs]
            [works-hub.pachyderm.util :refer (map->request-options map->str-ks)])
  (:import [clojure.lang IPersistentMap IPersistentStack]
           [com.stripe.model Customer]))

(defn create
  ([^IPersistentMap attrs]
   (create attrs nil))
  ([^IPersistentMap attrs ^IPersistentMap ro-opts]
   (trs/customer->map
    (Customer/create (map->str-ks attrs)
                     (map->request-options ro-opts)))))

(defn retrieve
  ([^String id]
   (retrieve id nil))
  ([^String id ^IPersistentMap ro-opts]
   (trs/customer->map
    (Customer/retrieve id
                       (map->request-options ro-opts)))))

(defn cancel-all-subscriptions
  ([^IPersistentMap customer]
   (cancel-all-subscriptions customer nil))
  ([^IPersistentMap customer ^IPersistentMap ro-opts]
   (if (not-empty (:subscriptions customer))
     (some->> customer
              :subscriptions
              (map #(subscriptions/cancel % {} ro-opts))
              doall)
     (throw (IllegalArgumentException.
             "provided customer has no subscriptions")))))

(defn update
  ([^IPersistentMap customer ^IPersistentMap attrs]
   (update customer attrs nil))
  ([^IPersistentMap customer ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ customer)]
     (trs/customer->map (.update o
                                 (map->str-ks attrs)
                                 (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "customers/update only accepts maps returned by customers/create, customers/retrieve, and customers/list")))))

(defn update-default-source
  ([^IPersistentMap customer ^String id]
   (update-default-source customer id nil))
  ([^IPersistentMap customer ^String id ^IPersistentMap ro-opts]
   (update customer {"default_source" id} ro-opts)))

(defn apply-coupon
  ([^IPersistentMap customer ^String coupon]
   (apply-coupon customer coupon nil))
  ([^IPersistentMap customer ^String coupon ^IPersistentMap ro-opts]
   (update customer {"coupon" coupon} ro-opts)))

(defn list
  ([]
   (list {}))
  ([^IPersistentMap attrs]
   (list attrs nil))
  ([^IPersistentMap attrs ^IPersistentMap ro-opts]
   (trs/customers-coll->seq
    (Customer/list
     (map->str-ks attrs)
     (map->request-options ro-opts)))))

(defn delete
  ([^IPersistentMap customer]
   (delete customer nil))
  ([^IPersistentMap customer ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ customer)]
     (trs/customer->map (.delete o (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "customers/delete only accepts maps returned by customers/create, customers/retrieve, and customers/list")))))

(defn list-subscriptions
  ([^IPersistentMap customer]
   (list-subscriptions customer {}))
  ([^IPersistentMap customer ^IPersistentMap attrs]
   (list-subscriptions customer attrs nil))
  ([^IPersistentMap customer ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if (:__origin__ customer)
     (subscriptions/list (merge attrs {:customer (:id customer)}) ro-opts)
     (throw (IllegalArgumentException.
             "customers/list-subscriptions only accepts maps returned by customers/create, customers/retrieve, and customers/list")))))

(defn retrieve-subscription
  ([^IPersistentMap customer ^String id]
   (retrieve-subscription customer id {}))
  ([^IPersistentMap customer ^String id ^IPersistentMap attrs]
   (retrieve-subscription customer id attrs nil))
  ([^IPersistentMap customer ^String id ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if (:__origin__ customer)
     (subscriptions/retrieve id (merge attrs {:customer (:id customer)}) ro-opts)
     (throw (IllegalArgumentException.
             "customers/list-subscriptions only accepts maps returned by customers/create, customers/retrieve, and customers/list")))))

;; aliases
(def subscribe subscriptions/create)
